<?php

namespace App\Http\Controllers\Accounting;

use App\Http\Controllers\Controller;
use App\Models\Currency;
use Illuminate\Http\Request;

class CurrencyController extends Controller
{
    public function index()
    {
        $currencies = Currency::all();
        return view('accounting.currencies.index', compact('currencies'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'code' => 'required|string|max:10|unique:currencies',
            'name' => 'required|string|max:255',
            'symbol' => 'required|string|max:10',
            'exchange_rate' => 'required|numeric|min:0.000001',
            'is_base' => 'boolean',
        ]);

        if ($validated['is_base'] ?? false) {
            Currency::where('is_base', true)->update(['is_base' => false]);
            $validated['exchange_rate'] = 1;
        }

        Currency::create($validated);

        return redirect()->route('accounting.currencies.index')
            ->with('success', 'Currency added successfully.');
    }

    public function update(Request $request, Currency $currency)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'symbol' => 'required|string|max:10',
            'exchange_rate' => 'required|numeric|min:0.000001',
            'is_base' => 'boolean',
            'is_active' => 'boolean',
        ]);

        if (($validated['is_base'] ?? false) && !$currency->is_base) {
            Currency::where('is_base', true)->update(['is_base' => false]);
            $validated['exchange_rate'] = 1;
        } elseif ($currency->is_base) {
            $validated['is_base'] = true; // Can't unset base unless another is set
            $validated['exchange_rate'] = 1;
        }

        $currency->update($validated);

        return redirect()->route('accounting.currencies.index')
            ->with('success', 'Currency updated successfully.');
    }

    public function destroy(Currency $currency)
    {
        if ($currency->is_base) {
            return back()->with('error', 'Cannot delete base currency.');
        }

        $currency->delete();
        return redirect()->route('accounting.currencies.index')
            ->with('success', 'Currency deleted successfully.');
    }
}
