<?php

namespace App\Http\Controllers\Accounting;

use App\Http\Controllers\Controller;
use App\Models\JournalEntry;
use App\Models\Account;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class JournalEntryController extends Controller
{
    public function index()
    {
        $entries = JournalEntry::with(['lines.account', 'user'])->latest()->paginate(15);
        return view('accounting.journals.index', compact('entries'));
    }

    public function create()
    {
        // Get all leaf accounts (no children)
        $accounts = Account::whereDoesntHave('children')->orderBy('code')->get();
        return view('accounting.journals.create', compact('accounts'));
    }

    use \App\Traits\HasLockCheck;

    public function store(Request $request)
    {
        $validated = $request->validate([
            'date' => 'required|date',
            'description' => 'nullable|string',
            'reference' => 'nullable|string',
            'lines' => 'required|array|min:2',
            'lines.*.account_id' => 'required|exists:accounts,id',
            'lines.*.debit' => 'required|numeric|min:0',
            'lines.*.credit' => 'required|numeric|min:0',
            'lines.*.description' => 'nullable|string',
        ]);

        if ($this->isDateLocked($validated['date'])) {
            return back()->with('error', 'The selected date is locked for accounting.')->withInput();
        }

        // Validate Debit == Credit
        $totalDebit = collect($validated['lines'])->sum('debit');
        $totalCredit = collect($validated['lines'])->sum('credit');

        if (abs($totalDebit - $totalCredit) > 0.01) {
            return back()->withInput()->withErrors(['lines' => "Journal Entry is not balanced! Debit: $totalDebit, Credit: $totalCredit"]);
        }

        DB::transaction(function () use ($validated, $totalDebit) {
            $entry = JournalEntry::create([
                'date' => $validated['date'],
                'description' => $validated['description'],
                'reference' => $validated['reference'],
                'is_manual' => true,
                'user_id' => auth()->id() ?? 1, // Fallback for dev if no auth
            ]);

            foreach ($validated['lines'] as $line) {
                // Skip lines with 0 debit AND 0 credit
                if ($line['debit'] == 0 && $line['credit'] == 0)
                    continue;

                $entry->lines()->create([
                    'account_id' => $line['account_id'],
                    'debit' => $line['debit'],
                    'credit' => $line['credit'],
                    'description' => $line['description'] ?? $validated['description'],
                ]);
            }
        });

        return redirect()->route('accounting.journals.index')
            ->with('success', 'Journal Entry posted successfully.');
    }

    public function show(JournalEntry $journal)
    {
        $journal->load(['lines.account', 'user']);
        return view('accounting.journals.show', compact('journal'));
    }
}
