<?php

namespace App\Http\Controllers\Accounting;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Project;
use App\Models\ProgrammerPayment;
use App\Services\FinancialTransactionService;
use Illuminate\Http\Request;

class ProgrammerController extends Controller
{
    protected $transactionService;

    public function __construct(FinancialTransactionService $transactionService)
    {
        $this->transactionService = $transactionService;
    }

    /**
     * Display a listing of the resource (Programmers).
     */
    public function index()
    {
        // Get users who are programmers (either by role or by being assigned to projects)
        // For now, let's get users who have projects assigned to them or have programmer payments
        $programmers = User::whereHas('projects')
            ->orWhereHas('roles', function ($q) {
                $q->where('name', 'programmer');
            })
            ->withCount('projects')
            ->paginate(15);

        return view('accounting.programmers.index', compact('programmers'));
    }

    /**
     * Display the specified resource (Programmer Account).
     */
    public function show(User $programmer)
    {
        $programmer->load(['projects.customer', 'programmerPayments.project', 'programmerPayments.recorder']);

        $stats = [
            'total_earned' => $programmer->total_earned,
            'total_paid' => $programmer->total_paid,
            'balance' => $programmer->balance,
            'projects_count' => $programmer->projects->count(),
        ];

        return view('accounting.programmers.show', compact('programmer', 'stats'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(User $programmer)
    {
        return view('accounting.programmers.edit', compact('programmer'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, User $programmer)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email,' . $programmer->id,
            'phone' => 'nullable|string|max:20',
        ]);

        $programmer->update($validated);

        return redirect()->route('accounting.programmers.show', $programmer)
            ->with('success', 'Programmer profile updated successfully.');
    }

    /**
     * Remove the specified resource from storage (with safety).
     */
    public function destroy(User $programmer)
    {
        if ($programmer->projects()->count() > 0) {
            return back()->with('error', 'Cannot delete programmer with assigned projects.');
        }

        $programmer->delete();

        return redirect()->route('accounting.programmers.index')
            ->with('success', 'Programmer deleted successfully.');
    }

    /**
     * Print Programmer Statement.
     */
    public function print(User $programmer)
    {
        $programmer->load(['projects.customer', 'programmerPayments.project']);
        return view('accounting.programmers.print', compact('programmer'));
    }
}
