<?php

namespace App\Http\Controllers\Accounting;

use App\Http\Controllers\Controller;
use App\Models\RecurringInvoice;
use App\Models\Customer;
use App\Models\Product;
use Illuminate\Http\Request;
use Carbon\Carbon;

class RecurringInvoiceController extends Controller
{
    public function index()
    {
        $recurringInvoices = RecurringInvoice::with('customer')->latest()->paginate(10);
        return view('accounting.recurring.index', compact('recurringInvoices'));
    }

    public function create()
    {
        $customers = Customer::where('status', 'active')->orderBy('name')->get();
        $products = Product::where('is_active', true)->orderBy('name')->get();
        return view('accounting.recurring.create', compact('customers', 'products'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'customer_id' => 'required|exists:customers,id',
            'title' => 'required|string|max:255',
            'frequency' => 'required|in:daily,weekly,monthly,quarterly,yearly',
            'start_date' => 'required|date',
            'items' => 'required|array|min:1',
            'items.*.description' => 'required|string',
            'items.*.quantity' => 'required|numeric|min:0.1',
            'items.*.unit_price' => 'required|numeric|min:0',
        ]);

        // Calculate total amount from items
        $total = 0;
        foreach ($validated['items'] as $item) {
            $total += $item['quantity'] * $item['unit_price'];
        }

        $recurring = new RecurringInvoice();
        $recurring->customer_id = $validated['customer_id'];
        $recurring->title = $validated['title'];
        $recurring->frequency = $validated['frequency'];
        $recurring->start_date = $validated['start_date'];
        $recurring->next_run_date = $validated['start_date']; // First run is start date
        $recurring->amount = $total;
        $recurring->items = $validated['items'];
        $recurring->notes = $request->notes;
        $recurring->is_active = true;
        $recurring->save();

        return redirect()->route('accounting.recurring.index')
            ->with('success', 'Subscription created successfully.');
    }

    public function show(RecurringInvoice $recurring)
    {
        return view('accounting.recurring.show', compact('recurring'));
    }

    public function edit(RecurringInvoice $recurring)
    {
        $customers = Customer::where('status', 'active')->orderBy('name')->get();
        $products = Product::where('is_active', true)->orderBy('name')->get();
        return view('accounting.recurring.edit', compact('recurring', 'customers', 'products'));
    }

    public function update(Request $request, RecurringInvoice $recurring)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'frequency' => 'required|in:daily,weekly,monthly,quarterly,yearly',
            'next_run_date' => 'required|date',
            'is_active' => 'boolean',
        ]);

        $recurring->update($validated);

        return redirect()->route('accounting.recurring.index')
            ->with('success', 'Subscription updated successfully.');
    }

    public function destroy(RecurringInvoice $recurring)
    {
        $recurring->delete();
        return redirect()->route('accounting.recurring.index')
            ->with('success', 'Subscription deleted.');
    }
}
