<?php

namespace App\Http\Controllers\Accounting;

use App\Http\Controllers\Controller;
use App\Models\Supplier;
use App\Models\SupplierService;
use App\Models\Currency;
use Illuminate\Http\Request;

class SupplierServiceController extends Controller
{
    public function index()
    {
        $services = SupplierService::with(['supplier', 'currency'])->latest()->get();
        $suppliers = Supplier::where('is_active', true)->get();
        $currencies = Currency::where('is_active', true)->get();

        return view('accounting.supplier_services.index', compact('services', 'suppliers', 'currencies'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'supplier_id' => 'required|exists:suppliers,id',
            'name' => 'required|string|max:255',
            'type' => 'required|in:reseller,server,domain,other',
            'cost_price' => 'required|numeric|min:0',
            'currency_id' => 'required|exists:currencies,id',
            'reseller_units' => 'required_if:type,reseller|nullable|integer|min:1',
            'markup_type' => 'required|in:fixed,percentage',
            'markup_value' => 'required|numeric',
            'notes' => 'nullable|string',
        ]);

        SupplierService::create($validated);

        return redirect()->route('accounting.supplier-services.index')
            ->with('success', 'Supplier service added successfully.');
    }

    public function update(Request $request, SupplierService $supplierService)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'type' => 'required|in:reseller,server,domain,other',
            'cost_price' => 'required|numeric|min:0',
            'currency_id' => 'required|exists:currencies,id',
            'reseller_units' => 'required_if:type,reseller|nullable|integer|min:1',
            'markup_type' => 'required|in:fixed,percentage',
            'markup_value' => 'required|numeric',
            'notes' => 'nullable|string',
            'is_active' => 'boolean',
        ]);

        $supplierService->update($validated);

        return redirect()->route('accounting.supplier-services.index')
            ->with('success', 'Supplier service updated successfully.');
    }

    public function destroy(SupplierService $supplierService)
    {
        $supplierService->delete();
        return redirect()->route('accounting.supplier-services.index')
            ->with('success', 'Supplier service deleted successfully.');
    }
}
