<?php

namespace App\Http\Controllers\Accounting;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

use App\Models\Customer;
use App\Models\Ticket;
use App\Models\TicketMessage;
use App\Services\WhmcsService;
use App\Services\WaSenderService;
use Illuminate\Support\Facades\DB;

class TicketController extends Controller
{
    protected $whmcs;
    protected $wasender;

    public function __construct(WhmcsService $whmcs, WaSenderService $wasender)
    {
        $this->whmcs = $whmcs;
        $this->wasender = $wasender;
    }

    public function index()
    {
        $tickets = Ticket::with('customer')->latest()->paginate(20);
        $customers = Customer::whereNotNull('whmcs_id')->get(); // Only those linked to support
        return view('accounting.tickets.index', compact('tickets', 'customers'));
    }

    public function storeGeneric(Request $request)
    {
        $request->validate([
            'customer_id' => 'required|exists:customers,id',
            'deptid' => 'required|integer',
            'subject' => 'required|string|max:255',
            'priority' => 'required|string',
            'message' => 'required|string',
        ]);

        $customer = Customer::find($request->customer_id);

        return $this->store($request, $customer);
    }

    public function store(Request $request, Customer $customer)
    {
        $request->validate([
            'deptid' => 'required|integer',
            'subject' => 'required|string|max:255',
            'priority' => 'required|string',
            'message' => 'required|string',
        ]);

        if (!$customer->whmcs_id) {
            return back()->with('error', 'العميل غير مربوط بـ WHMCS.');
        }

        $response = $this->whmcs->openTicket(
            $customer->whmcs_id,
            $request->deptid,
            $request->subject,
            $request->message,
            $request->priority
        );

        if (($response['result'] ?? '') !== 'success') {
            return back()->with('error', 'فشل في فتح التذكرة في WHMCS: ' . ($response['message'] ?? 'خطأ غير معروف'));
        }

        $deptNames = [
            1 => 'Support',
            2 => 'Sales',
            3 => 'Billing',
        ];

        $ticket = Ticket::create([
            'customer_id' => $customer->id,
            'whmcs_id' => $response['id'],
            'subject' => $request->subject,
            'status' => 'Open',
            'priority' => $request->priority,
            'department' => $deptNames[$request->deptid] ?? 'Support',
        ]);

        TicketMessage::create([
            'ticket_id' => $ticket->id,
            'user_id' => auth()->id(),
            'message' => $request->message,
            'is_admin' => true,
        ]);

        return back()->with('success', 'تم فتح التذكرة بنجاح ومزامنتها.');
    }

    public function sync(Customer $customer)
    {
        if (!$customer->whmcs_id) {
            return back()->with('error', 'Customer is not linked to WHMCS.');
        }

        $response = $this->whmcs->getClientTickets($customer->whmcs_id);

        if (($response['result'] ?? '') !== 'success') {
            return back()->with('error', 'Failed to fetch tickets: ' . ($response['message'] ?? 'Unknown error'));
        }

        $tickets = $response['tickets']['ticket'] ?? [];

        foreach ($tickets as $t) {
            Ticket::updateOrCreate(
                ['whmcs_id' => $t['id']],
                [
                    'customer_id' => $customer->id,
                    'subject' => $t['subject'],
                    'status' => $t['status'],
                    'priority' => $t['urgency'],
                    'department' => $t['deptname'],
                    'last_reply' => $t['lastreply'],
                ]
            );
        }

        return back()->with('success', 'Tickets synced successfully.');
    }

    public function show(Ticket $ticket)
    {
        $response = $this->whmcs->getTicketDetails($ticket->whmcs_id);
        $replies = $response['replies']['reply'] ?? [];

        // Optionally cache messages locally
        foreach ($replies as $r) {
            TicketMessage::updateOrCreate(
                ['ticket_id' => $ticket->id, 'created_at' => $r['date']],
                [
                    'message' => $r['message'],
                    'is_admin' => !empty($r['admin']),
                    'customer_id' => !empty($r['admin']) ? null : $ticket->customer_id,
                ]
            );
        }

        return view('accounting.tickets.show', compact('ticket', 'replies'));
    }

    public function reply(Request $request, Ticket $ticket)
    {
        $request->validate(['message' => 'required|string']);

        $response = $this->whmcs->addTicketReply($ticket->whmcs_id, $request->message);

        if (($response['result'] ?? '') !== 'success') {
            return back()->with('error', 'Failed to send reply to WHMCS.');
        }

        // Save locally
        TicketMessage::create([
            'ticket_id' => $ticket->id,
            'user_id' => auth()->id(),
            'message' => $request->message,
            'is_admin' => true,
        ]);

        // WhatsApp Notification
        if ($ticket->customer->whatsapp) {
            $msg = "📢 *Update on Ticket #{$ticket->whmcs_id}*\n\nSubject: {$ticket->subject}\n\nStatus: {$ticket->status}\n\nMessage: " . substr($request->message, 0, 100) . "...";
            $this->wasender->sendMessage($ticket->customer->whatsapp, $msg);
        }

        return back()->with('success', 'Reply sent and WhatsApp notification triggered.');
    }

    public function destroy(Ticket $ticket)
    {
        // Delete messages
        $ticket->messages()->delete();

        // Delete ticket
        $ticket->delete();

        return back()->with('success', __('Ticket deleted successfully.'));
    }
}

