<?php

namespace App\Http\Controllers\Accounting;

use App\Http\Controllers\Controller;
use App\Models\Unit;
use Illuminate\Http\Request;

class UnitController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $units = Unit::all();
        // If we want to group them or something, we can do it here.
        // For now just partial view or JSON for frontend components.
        return view('accounting.units.index', compact('units'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|unique:units,name',
            'short_name' => 'nullable|string',
            'type' => 'required|in:quantity,time',
            'parent_id' => 'nullable|exists:units,id',
            'conversion_factor' => 'required_with:parent_id|numeric',
        ]);

        $unit = Unit::create($validated);

        return redirect()->route('accounting.units.index')
            ->with('success', __('Unit created successfully.'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Unit $unit)
    {
        $validated = $request->validate([
            'name' => 'required|string|unique:units,name,' . $unit->id,
            'short_name' => 'nullable|string',
            'type' => 'required|in:quantity,time',
            'parent_id' => 'nullable|exists:units,id',
            'conversion_factor' => 'numeric',
        ]);

        $unit->update($validated);

        return redirect()->route('accounting.units.index')
            ->with('success', __('Unit updated successfully.'));
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Unit $unit)
    {
        // Check if used in products or invoices
        if ($unit->products()->count() > 0) {
            return back()->with('error', __('Cannot delete unit because it is assigned to products.'));
        }

        // Check invoice items manually if relation defined there
        // or just let DB restrict if foreign key is set (we set nullOnDelete so it is safe actually)

        $unit->delete();

        return redirect()->route('accounting.units.index')
            ->with('success', __('Unit deleted successfully.'));
    }
}
