<?php

namespace App\Http\Controllers;

use App\Models\Customer;
use App\Models\CustomerNote;
use App\Models\CustomerDocument;
use App\Services\WhmcsService;
use App\Services\WaSenderService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;

use App\Models\CustomerWalletTransaction;
use App\Models\LoyaltyTransaction;

class CustomerController extends Controller
{
    /**
     * Store loyalty transaction.
     */
    public function storeLoyaltyTransaction(Request $request, Customer $customer)
    {
        $validated = $request->validate([
            'points' => 'required|numeric|min:1',
            'type' => 'required|in:earned,redeemed',
            'description' => 'required|string',
            'date' => 'required|date',
        ]);

        // If redeeming, points should be negative or handled as deduction?
        // Observer increments strictly. So if redeeming, we should possibly store as negative?
        // View says: $transaction->points > 0 ? '+' : ''
        // And type badge.
        // Let's store negative points for redemption to be consistent with observer increment logic.
        // If user enters 50 points to redeem, we store -50.

        $points = $validated['points'];
        if ($validated['type'] === 'redeemed') {
            $points = -abs($points);
        } else {
            $points = abs($points);
        }

        $customer->loyaltyTransactions()->create([
            'points' => $points,
            'type' => $validated['type'],
            'description' => $validated['description'],
            'date' => $validated['date'],
            'user_id' => auth()->id(), // LoyaltyTransaction doesn't seem to have user_id in fillable? Checked in Step 8963.
            // checking fillable in step 8963: customer_id, points, type, reference_type, reference_id, description, date. NO user_id.
            // So I won't add user_id.
        ]);

        return back()->with('success', __('Loyalty transaction added successfully.'));
    }

    /**
     * Delete a loyalty transaction.
     */
    public function destroyLoyaltyTransaction(Customer $customer, LoyaltyTransaction $transaction)
    {
        if ($transaction->customer_id !== $customer->id) {
            abort(403);
        }

        $transaction->delete();

        return back()->with('success', __('Loyalty transaction deleted successfully.'));
    }

    /**
     * Update a loyalty transaction.
     */
    public function updateLoyaltyTransaction(Request $request, Customer $customer, LoyaltyTransaction $transaction)
    {
        if ($transaction->customer_id !== $customer->id) {
            abort(403);
        }

        $validated = $request->validate([
            'points' => 'required|numeric|min:1',
            'type' => 'required|in:earned,redeemed',
            'description' => 'required|string',
            'date' => 'required|date',
        ]);

        $points = $validated['points'];
        if ($validated['type'] === 'redeemed') {
            $points = -abs($points);
        } else {
            $points = abs($points);
        }

        $transaction->update([
            'points' => $points,
            'type' => $validated['type'],
            'description' => $validated['description'],
            'date' => $validated['date'],
        ]);

        return back()->with('success', __('Loyalty transaction updated successfully.'));
    }


    /**
     * Display a listing of customers.
     */
    public function index(Request $request)
    {
        $query = Customer::query();

        // Search
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                    ->orWhere('email', 'like', "%{$search}%")
                    ->orWhere('phone', 'like', "%{$search}%")
                    ->orWhere('company', 'like', "%{$search}%")
                    ->orWhere('customer_number', 'like', "%{$search}%");
            });
        }

        // Filter by status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Filter by customer type
        if ($request->filled('customer_type')) {
            $query->where('customer_type', $request->customer_type);
        }

        // Filter by source
        if ($request->filled('source')) {
            $query->where('source', $request->source);
        }

        // Stats
        $stats = [
            'total' => Customer::count(),
            'active' => Customer::where('status', 'active')->count(),
            'inactive' => Customer::where('status', 'inactive')->count(),
            'suspended' => Customer::where('status', 'suspended')->count(),
        ];

        // Sorting
        $sortField = $request->get('sort_by', 'created_at'); // Default sort by created_at
        $sortOrder = strtolower($request->get('sort_order', 'desc')) === 'asc' ? 'asc' : 'desc';

        $allowedSorts = ['name', 'created_at', 'customer_number'];

        if ($sortField === 'customer_number') {
            $query->orderByRaw('CAST(customer_number AS UNSIGNED) ' . $sortOrder);
        } elseif (in_array($sortField, $allowedSorts)) {
            $query->orderBy($sortField, $sortOrder);
        } else {
            $query->latest();
        }

        // Eager load group
        $customers = $query->with('customerGroup')->paginate(15)->withQueryString();

        return view('customers.index', compact('customers', 'stats'));
    }

    /**
     * Show the form for creating a new customer.
     */
    public function create()
    {
        $groups = \App\Models\CustomerGroup::all();
        return view('customers.create', compact('groups'));
    }

    /**
     * Store a newly created customer.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'nullable|email|unique:customers,email',
            'phone' => 'nullable|string|max:50',
            'whatsapp' => 'required|string|max:50',
            'opening_balance' => 'nullable|numeric',
            'company' => 'nullable|string|max:255',
            'address' => 'nullable|string',
            'city' => 'nullable|string|max:100',
            'country' => 'nullable|string|max:100',
            'status' => 'required|in:active,inactive,suspended',
            'customer_type' => 'required|in:individual,company',
            'notes' => 'nullable|string',
            'commercial_register' => 'nullable|string|max:255',
            'tax_id' => 'nullable|string|max:255',
            'customer_group_id' => 'nullable|exists:customer_groups,id',
            'loyalty_enabled' => 'boolean',
        ]);

        $validated['source'] = 'manual';
        $validated['balance'] = $validated['opening_balance'] ?? 0;

        $customer = Customer::create($validated);

        return redirect()->route('customers.show', $customer)
            ->with('success', __('messages.customer_created'));
    }

    /**
     * Display the specified customer.
     */
    public function show(Customer $customer)
    {
        $customer->load(['customerNotes.user', 'documents.uploader', 'customerGroup', 'loyaltyTransactions', 'invoices']);

        // Get WHMCS data if synced
        $whmcsData = null;
        if ($customer->isFromWhmcs()) {
            $whmcsService = new WhmcsService();
            if ($whmcsService->isConfigured()) {
                $whmcsData = [
                    'products' => $whmcsService->getClientProducts($customer->whmcs_id),
                    'domains' => $whmcsService->getClientDomains($customer->whmcs_id),
                    'invoices' => $whmcsService->getClientInvoices($customer->whmcs_id),
                    'tickets' => $whmcsService->getClientTickets($customer->whmcs_id),
                ];
            }
        }

        $treasuries = \App\Models\FinancialTreasury::where('is_active', true)->get();
        return view('customers.show', compact('customer', 'whmcsData', 'treasuries'));
    }

    /**
     * Show the form for editing the customer.
     */
    public function edit(Customer $customer)
    {
        $groups = \App\Models\CustomerGroup::all();
        return view('customers.edit', compact('customer', 'groups'));
    }

    /**
     * Update the specified customer.
     */
    public function update(Request $request, Customer $customer)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'nullable|email|unique:customers,email,' . $customer->id,
            'phone' => 'nullable|string|max:50',
            'whatsapp' => 'required|string|max:50',
            'company' => 'nullable|string|max:255',
            'address' => 'nullable|string',
            'city' => 'nullable|string|max:100',
            'country' => 'nullable|string|max:100',
            'status' => 'required|in:active,inactive,suspended',
            'customer_type' => 'required|in:individual,company',
            'notes' => 'nullable|string',
            'commercial_register' => 'nullable|string|max:255',
            'tax_id' => 'nullable|string|max:255',
            'customer_group_id' => 'nullable|exists:customer_groups,id',
            'loyalty_enabled' => 'boolean',
            'avatar' => 'nullable|image|max:2048',
        ]);

        // Fix checkbox handling
        $validated['loyalty_enabled'] = $request->has('loyalty_enabled');

        if ($request->hasFile('avatar')) {
            $path = $request->file('avatar')->store('avatars', 'public');
            $validated['avatar'] = $path;
        }

        $customer->update($validated);

        return redirect()->route('customers.show', $customer)
            ->with('success', __('messages.customer_updated'));
    }

    /**
     * Remove the specified customer.
     */
    public function destroy(Customer $customer)
    {
        $customer->delete();

        return redirect()->route('customers.index')
            ->with('success', __('messages.customer_deleted'));
    }

    /**
     * Sync customers from WHMCS.
     */
    public function syncWhmcs()
    {
        $whmcsService = new WhmcsService();

        if (!$whmcsService->isConfigured()) {
            return back()->with('error', __('messages.whmcs_not_configured'));
        }

        $result = $whmcsService->syncAllClients();

        if ($result['success']) {
            if ($result['synced'] > 0) {
                return back()->with('success', __('messages.whmcs_sync_success', ['count' => $result['synced']]));
            } elseif (!empty($result['errors'])) {
                // Success true but synced 0 and errors exist? Show errors.
                return back()->with('error', 'Sync completed with errors: ' . implode(', ', array_slice($result['errors'], 0, 3)));
            } else {
                return back()->with('info', __('messages.whmcs_sync_success', ['count' => 0]));
            }
        }

        return back()->with('error', $result['message']);
    }

    /**
     * Add a note to customer.
     */
    public function addNote(Request $request, Customer $customer)
    {
        $validated = $request->validate([
            'note' => 'required|string',
            'is_private' => 'boolean',
        ]);

        $customer->customerNotes()->create([
            'user_id' => auth()->id(),
            'note' => $validated['note'],
            'is_private' => $validated['is_private'] ?? false,
        ]);

        return back()->with('success', __('messages.note_added'));
    }

    /**
     * Upload document for customer.
     */
    public function uploadDocument(Request $request, Customer $customer)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'file' => 'required|file|max:10240', // 10MB max
        ]);

        $file = $request->file('file');
        $path = $file->store('customer-documents/' . $customer->id, 'public');

        $customer->documents()->create([
            'title' => $validated['title'],
            'file_path' => $path,
            'file_type' => $file->getClientOriginalExtension(),
            'file_size' => $file->getSize(),
            'uploaded_by' => auth()->id(),
        ]);

        return back()->with('success', __('messages.document_uploaded'));
    }

    /**
     * Send Portal Invitation via WhatsApp.
     */
    public function sendPortalInvite(Customer $customer)
    {
        // 1. Ensure token exists
        if (empty($customer->portal_token)) {
            $customer->generatePortalToken();
            $customer->save();
        }

        // 2. Fetch Template (mocking for now, or using a specific setting config)
        // Check for 'portal_welcome' template in Wa templates or Settings
        // For this implementation, we use a hardcoded template pattern or one from Settings if simple
        $template = "Hello {name},\n\nWelcome to our Client Portal! 🚀\nYou can track your services, invoices, and renewals here:\n{link}\n\nBest Regards,\nSMARTONE";

        // 3. Replace variables
        $message = str_replace(
            ['{name}', '{link}'],
            [$customer->name, $customer->portal_url],
            $template
        );

        // 4. Send Message using WaSender
        $phone = $customer->whatsapp ?: $customer->phone;

        if (!$phone) {
            return back()->with('error', __('messages.no_phone_number'));
        }

        $waSender = new WaSenderService();
        if (!$waSender->isConfigured()) {
            return back()->with('error', __('messages.wasender_not_configured'));
        }

        $result = $waSender->sendMessage($phone, $message);

        if ($result['success'] ?? false) {
            $customer->customerNotes()->create([
                'user_id' => auth()->id(),
                'note' => "🚀 Portal Invite Sent",
                'is_private' => false,
            ]);
            return back()->with('success', 'Portal invitation sent successfully!');
        }

        return back()->with('error', 'Failed to send WhatsApp: ' . ($result['message'] ?? 'Unknown error'));
    }

    /**
     * Send WhatsApp message to customer.
     */
    public function sendWhatsApp(Request $request, Customer $customer)
    {
        $validated = $request->validate([
            'message' => 'required|string',
        ]);

        $phone = $customer->whatsapp ?: $customer->phone;

        if (!$phone) {
            return back()->with('error', __('messages.no_phone_number'));
        }

        $waSender = new WaSenderService();

        if (!$waSender->isConfigured()) {
            return back()->with('error', __('messages.wasender_not_configured'));
        }

        $result = $waSender->sendMessage($phone, $validated['message']);

        if ($result['success'] ?? false) {
            // Log the message as a note
            $customer->customerNotes()->create([
                'user_id' => auth()->id(),
                'note' => "📱 WhatsApp: " . $validated['message'],
                'is_private' => false,
            ]);

            return back()->with('success', __('messages.whatsapp_sent'));
        }

        return back()->with('error', $result['message'] ?? __('messages.whatsapp_failed'));
    }

    /**
     * Export customers to CSV.
     */
    public function export(Request $request)
    {
        $customers = Customer::all();

        $filename = 'customers_' . date('Y-m-d_His') . '.csv';

        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => "attachment; filename=\"{$filename}\"",
        ];

        $callback = function () use ($customers) {
            $file = fopen('php://output', 'w');

            // Header row
            fputcsv($file, ['ID', 'Name', 'Email', 'Phone', 'Company', 'City', 'Country', 'Status', 'Type', 'Source', 'Created At']);

            foreach ($customers as $customer) {
                fputcsv($file, [
                    $customer->id,
                    $customer->name,
                    $customer->email,
                    $customer->phone,
                    $customer->company,
                    $customer->city,
                    $customer->country,
                    $customer->status,
                    $customer->customer_type,
                    $customer->source,
                    $customer->created_at->format('Y-m-d H:i:s'),
                ]);
            }

            fclose($file);
        };

    }

    /**
     * Store a wallet transaction for the customer.
     */
    public function storeWalletTransaction(Request $request, Customer $customer)
    {
        $validated = $request->validate([
            'type' => 'required|in:deposit,withdrawal,adjustment',
            'amount' => 'required|numeric|min:0.01',
            'payment_method' => 'nullable|string|max:100',
            'date' => 'required|date',
            'notes' => 'nullable|string',
            'treasury_id' => 'nullable|exists:financial_treasuries,id',
        ]);

        DB::transaction(function () use ($validated, $customer) {
            // 1. Create Wallet Transaction
            $walletTrx = $customer->walletTransactions()->create([
                'type' => $validated['type'],
                'amount' => $validated['amount'],
                'payment_method' => $validated['payment_method'],
                'date' => $validated['date'],
                'notes' => $validated['notes'],
                'user_id' => auth()->id(),
            ]);

            // 2. Handle Treasury Impact
            if (!empty($validated['treasury_id'])) {
                $transactionService = new \App\Services\FinancialTransactionService();
                $description = "Wallet {$validated['type']}: " . $customer->name . " - " . ($validated['notes'] ?? '');

                // Determine Category (Income vs Expense)
                // We need default categories or let user select? 
                // For now, let's hardcode or find 'Deposits'/'Withdrawals' category. 
                // Creating a helper or using ID 1 for now if not dynamic.
                // Ideally, we should have specific categories for 'Customer Deposits'.

                // Let's assume Category 1 = Income (Deposit), Category 2 = Expense (Withdrawal) based on previous search or robust logic.
                // A better way is to pass category_id too, but user didn't ask for it specifically in the modal.
                // I will try to find a category dynamically or use null if service allows (it requires int).
                // Let's use 1 for Income, 2 for Expense as placeholders or check DB.
                // In a real app we'd lookup 'Customer Deposits' category.

                $categoryId = 1; // Default

                if ($validated['type'] === 'deposit') {
                    // Income for Treasury
                    $transactionService->recordIncome(
                        $validated['amount'],
                        $validated['treasury_id'],
                        $categoryId,
                        $description,
                        $walletTrx,
                        null,
                        $validated['date']
                    );
                } elseif ($validated['type'] === 'withdrawal') {
                    // Expense for Treasury
                    $transactionService->recordExpense(
                        $validated['amount'],
                        $validated['treasury_id'],
                        $categoryId,
                        $description,
                        $walletTrx,
                        null,
                        $validated['date']
                    );
                }
                // 'adjustment' usually doesn't affect treasury unless specified.
            }
        });

        return back()->with('success', __('Wallet updated successfully.'));
    }
    /**
     * Update a customer note.
     */
    public function updateNote(Request $request, Customer $customer, CustomerNote $note)
    {
        // Ensure the note belongs to the customer
        if ($note->customer_id !== $customer->id) {
            abort(403);
        }

        $validated = $request->validate([
            'note' => 'required|string',
            'is_private' => 'boolean',
        ]);

        $note->update([
            'note' => $validated['note'],
            'is_private' => $validated['is_private'] ?? $note->is_private, // Preserve if not sent (e.g. inline edit might missed check)
        ]);

        return back()->with('success', __('Note updated successfully.'));
    }

    /**
     * Delete a customer note.
     */
    public function deleteNote(Customer $customer, CustomerNote $note)
    {
        if ($note->customer_id !== $customer->id) {
            abort(403);
        }

        $note->delete();

        return back()->with('success', __('Note deleted successfully.'));
    }

    /**
     * Delete a customer document.
     */
    public function deleteDocument(Customer $customer, CustomerDocument $document)
    {
        if ($document->customer_id !== $customer->id) {
            abort(403);
        }

        // Delete from storage
        if ($document->file_path && Storage::disk('public')->exists($document->file_path)) {
            Storage::disk('public')->delete($document->file_path);
        }

        $document->delete();

        return back()->with('success', __('Document deleted successfully.'));
    }
    /**
     * Delete a wallet transaction.
     */
    public function destroyWalletTransaction(Customer $customer, CustomerWalletTransaction $transaction)
    {
        if ($transaction->customer_id !== $customer->id) {
            abort(403);
        }

        DB::transaction(function () use ($transaction) {
            // Check for associated Financial Transaction
            $financialTrx = \App\Models\FinancialTransaction::where('model_type', get_class($transaction))
                ->where('model_id', $transaction->id)
                ->first();

            if ($financialTrx) {
                // Revert Treasury Balance manually since FinancialTransaction has no observer
                $treasury = $financialTrx->treasury;
                if ($treasury) {
                    if ($financialTrx->type == 'in') {
                        $treasury->decrement('balance', $financialTrx->amount);
                    } elseif ($financialTrx->type == 'out') {
                        $treasury->increment('balance', $financialTrx->amount);
                    }
                }
                $financialTrx->delete();
            }

            // Deleting the transaction will trigger the observer to reverse the balance change
            $transaction->delete();
        });

        return back()->with('success', __('Wallet transaction deleted successfully.'));
    }

    /**
     * Update a wallet transaction.
     */
    public function updateWalletTransaction(Request $request, Customer $customer, CustomerWalletTransaction $transaction)
    {
        if ($transaction->customer_id !== $customer->id) {
            abort(403);
        }

        $validated = $request->validate([
            'type' => 'required|in:deposit,withdrawal,adjustment',
            'amount' => 'required|numeric|min:0.01',
            'payment_method' => 'nullable|string|max:100',
            'date' => 'required|date',
            'notes' => 'nullable|string',
        ]);

        $transaction->update([
            'type' => $validated['type'],
            'amount' => $validated['amount'],
            'payment_method' => $validated['payment_method'],
            'date' => $validated['date'],
            'notes' => $validated['notes'],
        ]);

        return back()->with('success', __('Wallet transaction updated successfully.'));
    }
}

