<?php

namespace App\Http\Controllers\HR;

use App\Http\Controllers\Controller;
use App\Models\Attendance;
use App\Models\Employee;
use Illuminate\Http\Request;
use Carbon\Carbon;

class AttendanceController extends Controller
{
    public function index(Request $request)
    {
        $date = $request->input('date', now()->format('Y-m-d'));
        $employees = Employee::where('status', 'active')
            ->with([
                'attendanceLogs' => function ($q) use ($date) {
                    $q->whereDate('date', $date);
                }
            ])
            ->get();

        $attendances = Attendance::whereDate('date', $date)
            ->with('employee')
            ->get()
            ->keyBy('employee_id');

        return view('hr.attendance.index', compact('employees', 'attendances', 'date'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'employee_id' => 'required|exists:employees,id',
            'date' => 'required|date',
            'check_in' => 'nullable|date_format:H:i',
            'check_out' => 'nullable|date_format:H:i',
            'status' => 'required|in:present,late,absent,on_leave,half_day',
            'notes' => 'nullable|string',
        ]);

        // Auto-detect late status
        $status = $request->status;
        if ($request->check_in && $status === 'present') {
            $checkIn = Carbon::parse($request->check_in);
            $workStartTime = Carbon::parse('09:00'); // Default work start
            if ($checkIn->gt($workStartTime)) {
                $status = 'late';
            }
        }

        Attendance::updateOrCreate(
            [
                'employee_id' => $request->employee_id,
                'date' => $request->date,
            ],
            [
                'check_in' => $request->check_in,
                'check_out' => $request->check_out,
                'status' => $status,
                'notes' => $request->notes,
            ]
        );

        return redirect()->back()->with('success', __('Attendance recorded successfully.'));
    }

    public function bulkCheckIn(Request $request)
    {
        $request->validate([
            'employee_ids' => 'required|array',
            'employee_ids.*' => 'exists:employees,id',
        ]);

        $now = now();
        $workStartTime = Carbon::parse('09:00');
        $status = $now->format('H:i:s') > $workStartTime->format('H:i:s') ? 'late' : 'present';

        foreach ($request->employee_ids as $employeeId) {
            Attendance::updateOrCreate(
                [
                    'employee_id' => $employeeId,
                    'date' => $now->format('Y-m-d'),
                ],
                [
                    'check_in' => $now->format('H:i:s'),
                    'status' => $status,
                ]
            );
        }

        return redirect()->back()->with('success', __('Bulk check-in completed.'));
    }

    public function checkOut(Employee $employee)
    {
        $attendance = Attendance::where('employee_id', $employee->id)
            ->whereDate('date', now()->format('Y-m-d'))
            ->first();

        if ($attendance) {
            $attendance->update(['check_out' => now()->format('H:i:s')]);
            return redirect()->back()->with('success', __('Check-out recorded.'));
        }

        return redirect()->back()->with('error', __('No check-in found for today.'));
    }
}
