<?php

namespace App\Http\Controllers\HR;

use App\Http\Controllers\Controller;
use App\Models\Employee;
use Illuminate\Http\Request;

class EmployeeController extends Controller
{
    public function index()
    {
        $employees = Employee::latest()->paginate(15);
        $totalEmployees = Employee::count();
        $activeEmployees = Employee::active()->count();

        return view('hr.employees.index', compact('employees', 'totalEmployees', 'activeEmployees'));
    }

    public function create()
    {
        $departments = \App\Models\Department::where('is_active', true)->get();
        $designations = \App\Models\Designation::where('is_active', true)->orderBy('level')->get();
        return view('hr.employees.create', compact('departments', 'designations'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'first_name' => 'required|string|max:255',
            'last_name' => 'required|string|max:255',
            'email' => 'required|email|unique:employees,email',
            'phone' => 'nullable|string|max:20',
            // 'job_title' => 'required|string|max:255', // Deprecated in favor of designation_id
            'designation_id' => 'nullable|exists:designations,id',
            'department_id' => 'nullable|exists:departments,id',
            'basic_salary' => 'required|numeric|min:0',
            'joining_date' => 'nullable|date',
            'status' => 'required|in:active,on_leave,terminated,resigned',
            'shift_type' => 'nullable|string',
            'address' => 'nullable|string',
            'emergency_contact' => 'nullable|string',
            'notes' => 'nullable|string',
        ]);

        // Backward compatibility if needed, or just nullable
        $validated['job_title'] = $request->input('job_title', \App\Models\Designation::find($request->designation_id)?->name ?? 'Employee');
        $validated['department'] = \App\Models\Department::find($request->department_id)?->name ?? null;

        Employee::create($validated);

        return redirect()->route('hr.employees.index')->with('success', 'Employee added successfully.');
    }

    public function edit(Employee $employee)
    {
        $departments = \App\Models\Department::where('is_active', true)->get();
        $designations = \App\Models\Designation::where('is_active', true)->orderBy('level')->get();
        return view('hr.employees.edit', compact('employee', 'departments', 'designations'));
    }

    public function update(Request $request, Employee $employee)
    {
        $validated = $request->validate([
            'first_name' => 'required|string|max:255',
            'last_name' => 'required|string|max:255',
            'email' => 'required|email|unique:employees,email,' . $employee->id,
            'phone' => 'nullable|string|max:20',
            'designation_id' => 'nullable|exists:designations,id',
            'department_id' => 'nullable|exists:departments,id',
            'basic_salary' => 'required|numeric|min:0',
            'joining_date' => 'nullable|date',
            'status' => 'required|in:active,on_leave,terminated,resigned',
            'shift_type' => 'nullable|string',
            'address' => 'nullable|string',
            'emergency_contact' => 'nullable|string',
            'notes' => 'nullable|string',
        ]);

        $validated['job_title'] = $request->input('job_title', \App\Models\Designation::find($request->designation_id)?->name ?? $employee->job_title);
        $validated['department'] = \App\Models\Department::find($request->department_id)?->name ?? $employee->department;

        $employee->update($validated);

        return redirect()->route('hr.employees.index')->with('success', 'Employee updated successfully.');
    }

    public function destroy(Employee $employee)
    {
        $employee->delete();
        return back()->with('success', 'Employee deleted successfully.');
    }
}
