<?php

namespace App\Http\Controllers\HR;

use App\Http\Controllers\Controller;
use App\Models\Payroll;
use App\Models\PayrollItem;
use App\Models\Employee;
use App\Models\Treasury;
use Illuminate\Http\Request;
use Carbon\Carbon;

class PayrollController extends Controller
{
    public function index(Request $request)
    {
        $month = $request->input('month', now()->format('Y-m'));

        $payrolls = Payroll::where('period_month', $month)
            ->with('employee')
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        $employees = Employee::where('status', 'active')->get();
        $treasuries = Treasury::where('is_active', true)->get();

        return view('hr.payroll.index', compact('payrolls', 'month', 'employees', 'treasuries'));
    }

    public function generate(Request $request)
    {
        $request->validate([
            'period_month' => 'required|date_format:Y-m',
            'employee_ids' => 'nullable|array',
        ]);

        $month = $request->period_month;
        $employeeIds = $request->employee_ids;

        $query = Employee::where('status', 'active');
        if (!empty($employeeIds)) {
            $query->whereIn('id', $employeeIds);
        }

        $employees = $query->get();
        $generated = 0;

        foreach ($employees as $employee) {
            // Skip if payroll already exists
            $exists = Payroll::where('employee_id', $employee->id)
                ->where('period_month', $month)
                ->exists();

            if ($exists)
                continue;

            $payroll = Payroll::create([
                'employee_id' => $employee->id,
                'period_month' => $month,
                'basic_salary' => $employee->basic_salary ?? 0,
                'total_allowances' => 0,
                'total_deductions' => 0,
                'net_salary' => $employee->basic_salary ?? 0,
                'status' => 'draft',
            ]);

            $generated++;
        }

        return redirect()->route('hr.payroll.index', ['month' => $month])
            ->with('success', __(':count payroll records generated.', ['count' => $generated]));
    }

    public function show(Payroll $payroll)
    {
        $payroll->load('employee', 'items', 'treasury');
        return view('hr.payroll.show', compact('payroll'));
    }

    public function addItem(Request $request, Payroll $payroll)
    {
        if ($payroll->status === 'paid') {
            return redirect()->back()->with('error', __('Cannot modify a paid payroll.'));
        }

        $request->validate([
            'type' => 'required|in:allowance,deduction,bonus,penalty',
            'description' => 'required|string|max:255',
            'amount' => 'required|numeric|min:0',
        ]);

        PayrollItem::create([
            'payroll_id' => $payroll->id,
            'type' => $request->type,
            'description' => $request->description,
            'amount' => $request->amount,
        ]);

        // Recalculate
        $payroll->calculateNetSalary()->save();

        return redirect()->back()->with('success', __('Item added successfully.'));
    }

    public function removeItem(Payroll $payroll, PayrollItem $item)
    {
        if ($payroll->status === 'paid') {
            return redirect()->back()->with('error', __('Cannot modify a paid payroll.'));
        }

        $item->delete();
        $payroll->calculateNetSalary()->save();

        return redirect()->back()->with('success', __('Item removed.'));
    }

    public function approve(Payroll $payroll)
    {
        if ($payroll->status !== 'draft') {
            return redirect()->back()->with('error', __('Only draft payrolls can be approved.'));
        }

        $payroll->update(['status' => 'approved']);
        return redirect()->back()->with('success', __('Payroll approved.'));
    }

    public function pay(Request $request, Payroll $payroll)
    {
        if ($payroll->status !== 'approved') {
            return redirect()->back()->with('error', __('Only approved payrolls can be paid.'));
        }

        $request->validate([
            'treasury_id' => 'required|exists:treasuries,id',
        ]);

        // TODO: Integrate with Treasury transaction creation

        $payroll->update([
            'status' => 'paid',
            'payment_date' => now(),
            'treasury_id' => $request->treasury_id,
        ]);

        return redirect()->back()->with('success', __('Payroll marked as paid.'));
    }
}
