<?php

namespace App\Http\Controllers\Portal;

use App\Http\Controllers\Controller;
use App\Models\Customer;
use App\Models\Invoice;
use App\Models\Subscription;
use Illuminate\Http\Request;

class ClientPortalController extends Controller
{
    protected function getCustomer($token)
    {
        $customer = Customer::where('portal_token', $token)->firstOrFail();

        if ($customer->status !== 'active') {
            abort(403, 'Account is not active.');
        }

        return $customer;
    }

    public function index($token)
    {
        $customer = $this->getCustomer($token);

        // Stats
        $activeServices = Subscription::where('customer_id', $customer->id)->active()->count();
        $dueInvoices = Invoice::where('customer_id', $customer->id)->whereIn('status', ['unpaid', 'partially_paid'])->sum('total');
        $paidInvoices = Invoice::where('customer_id', $customer->id)->where('status', 'paid')->count();

        // Recent Invoices
        $recentInvoices = Invoice::where('customer_id', $customer->id)->latest()->take(5)->get();

        // Active Subscriptions
        $subscriptions = Subscription::where('customer_id', $customer->id)->active()->take(3)->get();

        return view('portal.dashboard', compact('customer', 'token', 'activeServices', 'dueInvoices', 'paidInvoices', 'recentInvoices', 'subscriptions'));
    }

    public function invoices($token)
    {
        $customer = $this->getCustomer($token);
        $invoices = Invoice::where('customer_id', $customer->id)->latest()->paginate(10);

        return view('portal.invoices', compact('customer', 'token', 'invoices'));
    }

    public function quotations($token)
    {
        $customer = $this->getCustomer($token);
        // Assuming Quotation model exists and has customer_id
        $quotations = \App\Models\Quotation::where('customer_id', $customer->id)->latest()->paginate(10);

        return view('portal.quotations.index', compact('customer', 'token', 'quotations'));
    }

    public function services($token)
    {
        $customer = $this->getCustomer($token);
        $subscriptions = Subscription::where('customer_id', $customer->id)->latest()->paginate(10);

        return view('portal.services', compact('customer', 'token', 'subscriptions'));
    }

    // Tickets
    public function tickets($token)
    {
        $customer = $this->getCustomer($token);
        $tickets = \App\Models\Ticket::where('customer_id', $customer->id)->latest()->paginate(10);
        return view('portal.tickets.index', compact('customer', 'token', 'tickets'));
    }

    public function createTicket($token)
    {
        $customer = $this->getCustomer($token);
        return view('portal.tickets.create', compact('customer', 'token'));
    }

    public function storeTicket(Request $request, $token, \App\Services\WhmcsService $whmcs)
    {
        $customer = $this->getCustomer($token);

        $request->validate([
            'subject' => 'required|string|max:255',
            'department_id' => 'required|integer',
            'priority' => 'required|string',
            'message' => 'required|string',
        ]);

        if (!$customer->whmcs_id) {
            return back()->with('error', 'Account not linked to support system.');
        }

        $response = $whmcs->openTicket(
            $customer->whmcs_id,
            $request->department_id,
            $request->subject,
            $request->message,
            $request->priority
        );

        if (($response['result'] ?? '') !== 'success') {
            return back()->with('error', 'Failed to open ticket: ' . ($response['message'] ?? 'Unknown Error'));
        }

        $deptNames = [
            1 => 'Support',
            2 => 'Sales',
            3 => 'Billing',
        ];

        $ticket = \App\Models\Ticket::create([
            'customer_id' => $customer->id,
            'whmcs_id' => $response['id'],
            'subject' => $request->subject,
            'status' => 'Open',
            'priority' => $request->priority,
            'department' => $deptNames[$request->department_id] ?? 'Support',
        ]);

        \App\Models\TicketMessage::create([
            'ticket_id' => $ticket->id,
            'customer_id' => $customer->id,
            'message' => $request->message,
            'is_admin' => false,
        ]);

        return redirect()->route('portal.tickets.show', ['token' => $token, 'ticket' => $ticket])
            ->with('success', 'Ticket created successfully.');
    }

    public function showTicket($token, \App\Models\Ticket $ticket, \App\Services\WhmcsService $whmcs)
    {
        $customer = $this->getCustomer($token);

        if ($ticket->customer_id !== $customer->id) {
            abort(403);
        }

        // Sync replies
        $response = $whmcs->getTicketDetails($ticket->whmcs_id);
        $replies = $response['replies']['reply'] ?? [];

        foreach ($replies as $r) {
            \App\Models\TicketMessage::updateOrCreate(
                ['ticket_id' => $ticket->id, 'created_at' => \Carbon\Carbon::parse($r['date'])],
                [
                    'message' => $r['message'],
                    'is_admin' => !empty($r['admin']),
                    'customer_id' => !empty($r['admin']) ? null : $customer->id,
                ]
            );
        }

        return view('portal.tickets.show', compact('customer', 'token', 'ticket', 'replies'));
    }

    public function replyTicket(Request $request, $token, \App\Models\Ticket $ticket, \App\Services\WhmcsService $whmcs)
    {
        $customer = $this->getCustomer($token);

        if ($ticket->customer_id !== $customer->id) {
            abort(403);
        }

        $request->validate(['message' => 'required|string']);

        $response = $whmcs->addTicketReply($ticket->whmcs_id, $request->message);

        if (($response['result'] ?? '') !== 'success') {
            return back()->with('error', 'Failed to send reply.');
        }

        \App\Models\TicketMessage::create([
            'ticket_id' => $ticket->id,
            'customer_id' => $customer->id,
            'message' => $request->message,
            'is_admin' => false,
        ]);

        return back()->with('success', 'Reply sent successfully.');
    }
}
