<?php

namespace App\Http\Controllers;

use App\Models\Product;
use App\Services\WhmcsService;
use Illuminate\Http\Request;

class ProductController extends Controller
{
    public function import(WhmcsService $whmcs)
    {
        $response = $whmcs->getProducts();
        $products = [];

        if (($response['result'] ?? '') == 'success') {
            $products = $response['products']['product'] ?? [];
            // Group by groupname
            $grouped = [];
            foreach ($products as $p) {
                $groupName = $p['groupname'] ?? 'General';
                if (!isset($grouped[$groupName])) {
                    $grouped[$groupName] = ['name' => $groupName, 'products' => ['product' => []]];
                }
                $grouped[$groupName]['products']['product'][] = $p;
            }
            $products = $grouped;
        }

        return view('accounting.products.import_whmcs', compact('products'));
    }

    public function storeImport(Request $request)
    {
        $validated = $request->validate([
            'selected_products' => 'required|array',
            'products' => 'required|array',
        ]);

        $count = 0;
        foreach ($validated['selected_products'] as $pid) {
            if (isset($validated['products'][$pid])) {
                $data = $validated['products'][$pid];

                \App\Models\Product::updateOrCreate(
                    ['name' => $data['name']], // Match by name to avoid duplicates
                    [
                        'description' => $data['description'],
                        'price' => floatval($data['price']),
                        'type' => 'service', // Default to service for WHMCS imports
                        'is_active' => true,
                    ]
                );
                $count++;
            }
        }

        return redirect()->route('products.index')->with('success', "$count products imported successfully.");
    }

    public function index()
    {
        $products = Product::where('is_active', true)->latest()->paginate(15);
        return view('products.index', compact('products'));
    }

    public function create()
    {
        $units = \App\Models\Unit::where('is_active', true)->get();
        return view('products.create', compact('units'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'type' => 'required|in:service,physical',
            'unit_id' => 'nullable|exists:units,id',
            'sku' => 'nullable|string|max:50|unique:products,sku',
            'price' => 'required|numeric|min:0',
            'price_usd' => 'nullable|numeric|min:0',
            'description' => 'nullable|string',
        ]);

        Product::create($validated);
        return redirect()->route('products.index')->with('success', 'Product created successfully.');
    }

    public function edit(Product $product)
    {
        $units = \App\Models\Unit::where('is_active', true)->get();
        return view('products.edit', compact('product', 'units'));
    }

    public function update(Request $request, Product $product)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'type' => 'required|in:service,physical',
            'unit_id' => 'nullable|exists:units,id',
            'sku' => 'nullable|string|max:50|unique:products,sku,' . $product->id,
            'price' => 'required|numeric|min:0',
            'description' => 'nullable|string',
        ]);

        $product->update($validated);
        return redirect()->route('products.index')->with('success', 'Product updated successfully.');
    }

    public function destroy(Product $product)
    {
        // Ideally just deactivate, but for now delete
        $product->delete();
        return back()->with('success', 'Product deleted successfully.');
    }
}
