<?php

namespace App\Http\Controllers;

use App\Models\Setting;
use App\Models\EmailTemplate;
use App\Services\WhmcsService;
use App\Services\WaSenderService;
use App\Services\EmailService;
use Illuminate\Http\Request;

class SettingController extends Controller
{
    /**
     * Display settings page.
     */
    public function index()
    {
        $settings = [
            'general' => Setting::getGroup('general'),
            'whmcs' => Setting::getGroup('whmcs'),
            'wasender' => Setting::getGroup('wasender'),
            'smtp' => Setting::getGroup('smtp'),
            'security' => Setting::getGroup('security'),
        ];

        $emailTemplates = EmailTemplate::all();

        return view('settings.index', compact('settings', 'emailTemplates'));
    }

    /**
     * Update general settings.
     */
    public function updateGeneral(Request $request)
    {
        $validated = $request->validate([
            'company_name' => 'required|string|max:255',
            'company_email' => 'nullable|email|max:255',
            'company_phone' => 'nullable|string|max:50',
            'company_website' => 'nullable|url|max:255',
            'company_address' => 'nullable|string',
            'timezone' => 'nullable|string|max:100',
            'default_language' => 'required|in:en,ar',
            'company_logo' => 'nullable|image|max:2048', // Max 2MB
            'accounting_lock_date' => 'nullable|date',
            'expected_annual_revenue' => 'nullable|numeric|min:0',
        ]);

        // Handle File Upload
        if ($request->hasFile('company_logo')) {
            $path = $request->file('company_logo')->store('uploads/settings', 'public');
            $validated['company_logo'] = $path;
        }

        Setting::setGroup('general', $validated);

        // Security Settings (handled within general form for convenience)
        Setting::setValue('security', 'enable_otp', $request->boolean('enable_otp'));

        return back()->with('success', __('messages.settings_updated'));
    }

    /**
     * Close accounting month by setting lock date.
     */
    public function closeMonth(Request $request)
    {
        $validated = $request->validate([
            'lock_date' => 'required|date',
        ]);

        Setting::setValue('general', 'accounting_lock_date', $validated['lock_date']);

        return back()->with('success', __('Accounting period locked successfully up to ') . $validated['lock_date']);
    }

    /**
     * Update WHMCS settings.
     */
    public function updateWhmcs(Request $request)
    {
        $validated = $request->validate([
            'api_url' => 'nullable|url|max:500',
            'api_identifier' => 'nullable|string|max:255',
            'api_secret' => 'nullable|string|max:255',
            'auto_sync' => 'boolean',
        ]);

        Setting::setGroup('whmcs', $validated);

        return back()->with('success', __('messages.whmcs_settings_updated'));
    }

    /**
     * Test WHMCS connection.
     */
    public function testWhmcs()
    {
        $whmcsService = new WhmcsService();
        $result = $whmcsService->testConnection();

        return response()->json($result);
    }

    /**
     * Update WaSender settings.
     */
    public function updateWaSender(Request $request)
    {
        $validated = $request->validate([
            'api_url' => 'nullable|url|max:500',
            'api_key' => 'nullable|string|max:255',
            'session_id' => 'nullable|string|max:255',
        ]);

        Setting::setGroup('wasender', $validated);

        return back()->with('success', __('messages.wasender_settings_updated'));
    }

    /**
     * Test WaSender connection.
     */
    public function testWaSender()
    {
        $waSenderService = new WaSenderService();
        $result = $waSenderService->getSessionStatus();

        return response()->json([
            'success' => isset($result['status']),
            'status' => $result['status'] ?? 'unknown',
            'message' => $result['message'] ?? 'Session status retrieved',
        ]);
    }

    /**
     * Update SMTP settings.
     */
    public function updateSmtp(Request $request)
    {
        $validated = $request->validate([
            'host' => 'nullable|string|max:255',
            'port' => 'nullable|integer|min:1|max:65535',
            'username' => 'nullable|string|max:255',
            'password' => 'nullable|string|max:255',
            'encryption' => 'nullable|in:tls,ssl,null',
            'from_address' => 'nullable|email|max:255',
            'from_name' => 'nullable|string|max:255',
        ]);

        Setting::setGroup('smtp', $validated);

        return back()->with('success', __('messages.smtp_settings_updated'));
    }

    /**
     * Test SMTP connection.
     */
    public function testSmtp()
    {
        $emailService = new EmailService();
        $result = $emailService->testConnection();

        return response()->json($result);
    }

    /**
     * Store a new email template.
     */
    public function storeTemplate(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255|unique:email_templates,name',
            'subject' => 'required|string|max:500',
            'body' => 'required|string',
            'type' => 'nullable|string|max:100',
            'variables' => 'nullable|array',
        ]);

        EmailTemplate::create($validated);

        return back()->with('success', __('messages.template_created'));
    }

    /**
     * Update an email template.
     */
    public function updateTemplate(Request $request, EmailTemplate $template)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255|unique:email_templates,name,' . $template->id,
            'subject' => 'required|string|max:500',
            'body' => 'required|string',
            'type' => 'nullable|string|max:100',
            'variables' => 'nullable|array',
            'is_active' => 'boolean',
        ]);

        $template->update($validated);

        return back()->with('success', __('messages.template_updated'));
    }

    /**
     * Delete an email template.
     */
    public function destroyTemplate(EmailTemplate $template)
    {
        $template->delete();

        return back()->with('success', __('messages.template_deleted'));
    }
}
