<?php

namespace App\Http\Controllers\Settings;

use App\Http\Controllers\Controller;
use Spatie\Permission\Models\Role;
use Spatie\Permission\Models\Permission;
use Illuminate\Http\Request;

class RoleController extends Controller
{
    public function index()
    {
        $roles = Role::with('permissions')->get();
        return view('settings.roles.index', compact('roles'));
    }

    public function create()
    {
        $permissions = Permission::all();
        return view('settings.roles.create', compact('permissions'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255|unique:roles',
            'permissions' => 'array'
        ]);

        $role = Role::create(['name' => $request->name]);

        if ($request->has('permissions')) {
            $role->syncPermissions($request->permissions);
        }

        return redirect()->route('settings.roles.index')->with('success', __('Role created successfully.'));
    }

    public function edit(Role $role)
    {
        $permissions = Permission::all();

        $modules = [
            'Customers' => 'customers',
            'Companies' => 'companies',
            'Transactions' => 'transactions',
            'Sales' => 'sales',
            'Bank & Cash' => 'treasuries',
            'Products & Services' => 'products',
            'Reports' => 'reports',
            'Utilities' => 'utilities',
            'Appearance' => 'appearance',
            'Plugins' => 'plugins',
            'Calendar' => 'calendar',
            'Leads' => 'leads',
            'Tasks' => 'tasks',
            'Contracts' => 'contracts',
            'Orders' => 'orders',
            'Settings' => 'settings',
            'Documents' => 'documents',
        ];

        $actions = [
            'view' => 'عرض',
            'create' => 'خلق',
            'edit' => 'تعديل',
            'delete' => 'حذف',
            'full' => 'كل المعلومات',
        ];

        return view('settings.roles.edit', compact('role', 'permissions', 'modules', 'actions'));
    }

    public function update(Request $request, Role $role)
    {
        $request->validate([
            'name' => 'required|string|max:255|unique:roles,name,' . $role->id,
            'permissions' => 'array'
        ]);

        $role->update(['name' => $request->name]);
        $role->syncPermissions($request->permissions ?? []);

        return redirect()->route('settings.roles.index')->with('success', __('Role updated successfully.'));
    }

    public function destroy(Role $role)
    {
        if ($role->name === 'admin') {
            return back()->with('error', __('The admin role cannot be deleted.'));
        }
        $role->delete();
        return redirect()->route('settings.roles.index')->with('success', __('Role deleted successfully.'));
    }
}
