<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class Customer extends Model
{
    use HasFactory, SoftDeletes, \App\Traits\HasOwnership;

    protected $fillable = [
        'whmcs_id',
        'name',
        'email',
        'phone',
        'whatsapp',
        'company',
        'address',
        'city',
        'country',
        'status',
        'customer_type',
        'source',
        'notes',
        'portal_token',
        'portal_token_created_at',
        'customer_group_id',
        'loyalty_enabled',
        'loyalty_points',
        'customer_number',
        'commercial_register',
        'tax_id',
        'opening_balance',
        'balance',
        'avatar',
    ];

    protected $casts = [
        'whmcs_id' => 'integer',
        'portal_token_created_at' => 'datetime',
        'loyalty_enabled' => 'boolean',
        'loyalty_points' => 'decimal:2',
        'opening_balance' => 'decimal:2',
        'balance' => 'decimal:2',
    ];

    public static function boot()
    {
        parent::boot();

        static::creating(function ($customer) {
            if (empty($customer->portal_token)) {
                $customer->generatePortalToken();
            }

            // Generate Customer Number
            if (empty($customer->customer_number)) {
                if ($customer->whmcs_id) {
                    $customer->customer_number = (string) $customer->whmcs_id;
                } else {
                    $lastNumber = static::whereNotNull('customer_number')
                        ->whereRaw('CAST(customer_number AS UNSIGNED) >= 10000')
                        ->orderByRaw('CAST(customer_number AS UNSIGNED) DESC')
                        ->first();

                    $nextNumber = $lastNumber ? (intval($lastNumber->customer_number) + 1) : 10000;
                    $customer->customer_number = (string) $nextNumber;
                }
            }
        });
    }

    public function generatePortalToken()
    {
        $this->portal_token = \Illuminate\Support\Str::random(60);
        $this->portal_token_created_at = now();
    }

    public function getPortalUrlAttribute()
    {
        return route('portal.index', $this->portal_token);
    }

    public function customerGroup(): BelongsTo
    {
        return $this->belongsTo(CustomerGroup::class);
    }

    public function loyaltyTransactions(): HasMany
    {
        return $this->hasMany(LoyaltyTransaction::class);
    }

    public function walletTransactions(): HasMany
    {
        return $this->hasMany(CustomerWalletTransaction::class);
    }

    /**
     * Get the notes for the customer.
     */
    public function customerNotes(): HasMany
    {
        return $this->hasMany(CustomerNote::class);
    }

    public function projects(): HasMany
    {
        return $this->hasMany(Project::class);
    }

    public function invoices(): HasMany
    {
        return $this->hasMany(Invoice::class);
    }

    public function licenses(): HasMany
    {
        return $this->hasMany(SoftwareLicense::class);
    }

    /**
     * Get the documents for the customer.
     */
    public function documents(): HasMany
    {
        return $this->hasMany(CustomerDocument::class);
    }

    /**
     * Scope for active customers
     */
    public function scopeActive($query)
    {
        return $query->where('status', 'active');
    }

    /**
     * Scope for customers from WHMCS
     */
    public function scopeFromWhmcs($query)
    {
        return $query->whereNotNull('whmcs_id');
    }

    /**
     * Check if customer is synced from WHMCS
     */
    public function isFromWhmcs(): bool
    {
        return !is_null($this->whmcs_id);
    }

    /**
     * Get display name (company or name)
     */
    public function getDisplayNameAttribute(): string
    {
        return $this->company ?: $this->name;
    }

    /**
     * Get status badge color
     */
    public function getStatusColorAttribute(): string
    {
        return match ($this->status) {
            'active' => 'success',
            'inactive' => 'warning',
            'suspended' => 'danger',
            default => 'secondary',
        };
    }
    /**
     * Get Total Balance (Invoices - Payments) in Base Currency.
     */
    /**
     * Get Total Debt (Unpaid/Partial Invoices) in Base Currency.
     */
    public function getTotalDebtAttribute()
    {
        $invoices = $this->invoices()->whereIn('status', ['unpaid', 'partial', 'sent'])->get();
        $totalDebtEgp = 0;

        foreach ($invoices as $invoice) {
            $remaining = $invoice->total - $invoice->paid_amount;
            $rate = $invoice->exchange_rate ?: 1;
            $totalDebtEgp += $remaining * $rate;
        }

        return $totalDebtEgp;
    }

    /**
     * Get Wallet Balance in Base Currency.
     */
    public function getWalletBalanceAttribute()
    {
        return $this->balance;
    }

    /**
     * Get Total Balance (Debt - Wallet) in Base Currency.
     * Positive means they owe us. Negative means we owe them (Credit).
     */
    public function getBaseCurrencyBalanceAttribute()
    {
        return $this->total_debt - $this->wallet_balance;
    }

    /**
     * Get Estimated Balance in USD.
     */
    public function getUsdBalanceAttribute()
    {
        $baseBalance = $this->base_currency_balance;

        // Find USD currency to get rate
        $usd = Currency::where('code', 'USD')->first();
        $rate = ($usd && $usd->exchange_rate > 0) ? $usd->exchange_rate : 1;

        // Convert EGP to USD
        // 50 EGP / 50 Rate = 1 USD
        return $baseBalance / $rate;
    }
}
