<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class SupplierService extends Model
{
    protected $fillable = [
        'supplier_id',
        'name',
        'type',
        'cost_price',
        'currency_id',
        'reseller_units',
        'markup_type',
        'markup_value',
        'notes',
        'is_active',
    ];

    protected $casts = [
        'is_active' => 'boolean',
        'cost_price' => 'decimal:2',
        'markup_value' => 'decimal:2',
    ];

    public function supplier(): BelongsTo
    {
        return $this->belongsTo(Supplier::class);
    }

    public function currency(): BelongsTo
    {
        return $this->belongsTo(Currency::class);
    }

    /**
     * Calculate the base cost per unit (especially for resellers).
     */
    public function getBaseCostAttribute()
    {
        if ($this->type === 'reseller' && $this->reseller_units > 0) {
            return $this->cost_price / $this->reseller_units;
        }
        return $this->cost_price;
    }

    /**
     * Calculate suggested selling price based on markup logic.
     */
    public function getSuggestedPriceAttribute()
    {
        $base = $this->base_cost;

        if ($this->markup_type === 'percentage') {
            return $base + ($base * $this->markup_value / 100);
        }

        return $base + $this->markup_value;
    }
}
