<?php

namespace App\Services;

use App\Models\Setting;
use App\Models\EmailTemplate;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;
use Illuminate\Mail\Message;

class EmailService
{
    protected array $config;

    public function __construct()
    {
        $this->loadConfig();
    }

    /**
     * Load SMTP configuration from settings
     */
    protected function loadConfig(): void
    {
        $this->config = [
            'host' => Setting::getValue('smtp', 'host', config('mail.mailers.smtp.host')),
            'port' => Setting::getValue('smtp', 'port', config('mail.mailers.smtp.port')),
            'username' => Setting::getValue('smtp', 'username', config('mail.mailers.smtp.username')),
            'password' => Setting::getValue('smtp', 'password', config('mail.mailers.smtp.password')),
            'encryption' => Setting::getValue('smtp', 'encryption', config('mail.mailers.smtp.encryption')),
            'from_address' => Setting::getValue('smtp', 'from_address', config('mail.from.address')),
            'from_name' => Setting::getValue('smtp', 'from_name', config('mail.from.name')),
        ];

        // Dynamically update mail config
        config([
            'mail.mailers.smtp.host' => $this->config['host'],
            'mail.mailers.smtp.port' => $this->config['port'],
            'mail.mailers.smtp.username' => $this->config['username'],
            'mail.mailers.smtp.password' => $this->config['password'],
            'mail.mailers.smtp.encryption' => $this->config['encryption'],
            'mail.from.address' => $this->config['from_address'],
            'mail.from.name' => $this->config['from_name'],
        ]);
    }

    /**
     * Check if email is configured
     */
    public function isConfigured(): bool
    {
        return !empty($this->config['host']) &&
            !empty($this->config['username']) &&
            !empty($this->config['from_address']);
    }

    /**
     * Test SMTP connection
     */
    public function testConnection(): array
    {
        try {
            $transport = new \Symfony\Component\Mailer\Transport\Smtp\EsmtpTransport(
                $this->config['host'],
                (int) $this->config['port'],
                $this->config['encryption'] === 'tls'
            );

            $transport->setUsername($this->config['username']);
            $transport->setPassword($this->config['password']);

            // Try to start the transport
            $transport->start();
            $transport->stop();

            return ['success' => true, 'message' => 'SMTP connection successful'];
        } catch (\Exception $e) {
            Log::error('SMTP Test Error: ' . $e->getMessage());
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }

    /**
     * Send email using template
     */
    public function send(string $to, string $templateName, array $data = []): bool
    {
        try {
            $template = EmailTemplate::where('name', $templateName)->active()->first();

            if (!$template) {
                Log::error("Email template not found: {$templateName}");
                return false;
            }

            $parsed = $template->parse($data);

            Mail::raw($parsed['body'], function (Message $message) use ($to, $parsed) {
                $message->to($to)
                    ->subject($parsed['subject']);
            });

            return true;
        } catch (\Exception $e) {
            Log::error('Email Send Error: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Send bulk emails
     */
    public function sendBulk(array $recipients, string $templateName, array $data = []): array
    {
        $results = [];

        foreach ($recipients as $email) {
            $results[$email] = $this->send($email, $templateName, $data);
        }

        return $results;
    }

    /**
     * Send raw email (without template)
     */
    public function sendRaw(string $to, string $subject, string $body): bool
    {
        try {
            Mail::raw($body, function (Message $message) use ($to, $subject) {
                $message->to($to)
                    ->subject($subject);
            });

            return true;
        } catch (\Exception $e) {
            Log::error('Email Send Error: ' . $e->getMessage());
            return false;
        }
    }
}
