<?php

namespace App\Services;

use App\Models\Setting;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class WaSenderService
{
    protected string $apiUrl;
    protected string $apiKey;
    protected string $sessionId;

    public function __construct()
    {
        $this->apiUrl = Setting::getValue('wasender', 'api_url', config('services.wasender.url', 'https://api.wasenderapi.com'));
        $this->apiKey = Setting::getValue('wasender', 'api_key', config('services.wasender.api_key', ''));
        $this->sessionId = Setting::getValue('wasender', 'session_id', config('services.wasender.session_id', ''));
    }

    /**
     * Check if WaSender is configured
     */
    public function isConfigured(): bool
    {
        return !empty($this->apiKey) && !empty($this->sessionId);
    }

    /**
     * Make API request to WaSender
     */
    protected function makeRequest(string $endpoint, string $method = 'POST', array $data = []): array
    {
        if (!$this->isConfigured()) {
            return ['success' => false, 'message' => 'WaSender is not configured'];
        }

        try {
            $response = Http::withHeaders([
                        'Authorization' => 'Bearer ' . $this->apiKey,
                        'Content-Type' => 'application/json',
                    ])->{strtolower($method)}($this->apiUrl . $endpoint, $data);

            return $response->json() ?? ['success' => false, 'message' => 'Invalid response'];
        } catch (\Exception $e) {
            Log::error('WaSender API Error: ' . $e->getMessage());
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }

    /**
     * Get session status
     */
    public function getSessionStatus(): array
    {
        return $this->makeRequest('/sessions/' . $this->sessionId, 'GET');
    }

    /**
     * Send text message
     */
    public function sendMessage(string $phone, string $message): array
    {
        // Normalize phone number (remove spaces, dashes, etc.)
        $phone = preg_replace('/[^0-9]/', '', $phone);

        // Add country code if not present (assuming Egyptian numbers)
        if (!str_starts_with($phone, '20') && strlen($phone) === 11) {
            $phone = '20' . substr($phone, 1);
        }

        return $this->makeRequest('/messages/send', 'POST', [
            'sessionId' => $this->sessionId,
            'to' => $phone,
            'text' => $message,
        ]);
    }

    /**
     * Send message with template
     */
    public function sendWhatsAppTemplate(string $phone, string $key, array $data = []): array
    {
        $template = \App\Models\WhatsAppTemplate::where('key', $key)->where('is_active', true)->first();

        if (!$template) {
            return ['success' => false, 'message' => 'Template not found or inactive'];
        }

        $message = $template->parse($data);

        return $this->sendMessage($phone, $message);
    }

    /**
     * Send media message
     */
    public function sendMedia(string $phone, string $mediaUrl, string $caption = ''): array
    {
        $phone = preg_replace('/[^0-9]/', '', $phone);

        if (!str_starts_with($phone, '20') && strlen($phone) === 11) {
            $phone = '20' . substr($phone, 1);
        }

        return $this->makeRequest('/messages/send-media', 'POST', [
            'sessionId' => $this->sessionId,
            'to' => $phone,
            'mediaUrl' => $mediaUrl,
            'caption' => $caption,
        ]);
    }

    /**
     * Send bulk messages
     */
    public function sendBulk(array $recipients, string $message): array
    {
        $results = [];

        foreach ($recipients as $phone) {
            $results[$phone] = $this->sendMessage($phone, $message);
            // Add small delay to avoid rate limiting
            usleep(500000); // 0.5 seconds
        }

        return $results;
    }
}
