<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration {
    public function up(): void
    {
        // 1. Create units table
        Schema::create('units', function (Blueprint $table) {
            $table->id();
            $table->string('name')->unique(); // Kilogram, Monthly
            $table->string('short_name')->nullable(); // kg, mo
            $table->enum('type', ['quantity', 'time'])->default('quantity');
            $table->boolean('is_active')->default(true);
            $table->unsignedBigInteger('parent_id')->nullable(); // For sub-units (e.g., Gram under Kilogram)
            $table->foreign('parent_id')->references('id')->on('units')->nullOnDelete();
            $table->decimal('conversion_factor', 10, 4)->default(1); // 1kg = 1000g
            $table->timestamps();
        });

        // 2. Add unit_id to products
        Schema::table('products', function (Blueprint $table) {
            $table->foreignId('unit_id')->nullable()->after('type')->constrained('units')->nullOnDelete();
        });

        // 3. Add unit_id + dates to invoice_items
        Schema::table('invoice_items', function (Blueprint $table) {
            $table->foreignId('unit_id')->nullable()->after('description')->constrained('units')->nullOnDelete();
            $table->date('start_date')->nullable()->after('total');
            $table->date('end_date')->nullable()->after('start_date');
        });

        // 4. Add unit_id + dates to quotation_items
        Schema::table('quotation_items', function (Blueprint $table) {
            $table->foreignId('unit_id')->nullable()->after('description')->constrained('units')->nullOnDelete();
            $table->date('start_date')->nullable()->after('total');
            $table->date('end_date')->nullable()->after('start_date');
        });
    }

    public function down(): void
    {
        Schema::table('quotation_items', function (Blueprint $table) {
            $table->dropForeign(['unit_id']);
            $table->dropColumn(['unit_id', 'start_date', 'end_date']);
        });

        Schema::table('invoice_items', function (Blueprint $table) {
            $table->dropForeign(['unit_id']);
            $table->dropColumn(['unit_id', 'start_date', 'end_date']);
        });

        Schema::table('products', function (Blueprint $table) {
            $table->dropForeign(['unit_id']);
            $table->dropColumn('unit_id');
        });

        Schema::dropIfExists('units');
    }
};
