<?php

/**
 * SMARTONE ERP - Professional Web Installer
 * Handles requirements check, database configuration, and system setup.
 */

session_start();

// Define paths
define('LARAVEL_START', microtime(true));
$basePath = realpath(__DIR__ . '/..');
$envPath = $basePath . '/.env';

// System Requirements
$requirements = [
    'php' => '8.2.0',
    'extensions' => [
        'bcmath', 'ctype', 'fileinfo', 'mbstring', 'openssl', 'pdo', 'tokenizer', 'xml', 'pdo_mysql'
    ],
    'directories' => [
        'storage' => 0775,
        'bootstrap/cache' => 0775,
    ]
];

$step = isset($_GET['step']) ? (int)$_GET['step'] : 1;
$errors = [];

// Helper to check PHP version
function checkPHPVersion($min) {
    return version_compare(PHP_VERSION, $min, '>=');
}

// Helper to check extensions
function checkExtensions($exts) {
    $results = [];
    foreach ($exts as $ext) {
        $results[$ext] = extension_loaded($ext);
    }
    return $results;
}

// Helper to check permissions
function checkPermissions($dirs, $basePath) {
    $results = [];
    foreach ($dirs as $dir => $perm) {
        $fullPath = $basePath . '/' . $dir;
        $results[$dir] = is_dir($fullPath) && is_writable($fullPath);
    }
    return $results;
}

// Installation Logic
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if ($step === 2) {
        // Test Database Connection
        $dbHost = $_POST['db_host'];
        $dbName = $_POST['db_name'];
        $dbUser = $_POST['db_user'];
        $dbPass = $_POST['db_pass'];

        try {
            $dsn = "mysql:host=$dbHost;dbname=$dbName;charset=utf8mb4";
            $pdo = new PDO($dsn, $dbUser, $dbPass, [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]);
            
            $_SESSION['db_config'] = [
                'host' => $dbHost,
                'name' => $dbName,
                'user' => $dbUser,
                'pass' => $dbPass
            ];
            header('Location: install.php?step=3');
            exit;
        } catch (PDOException $e) {
            $errors[] = "Database Connection Failed: " . $e->getMessage();
        }
    } elseif ($step === 3) {
        // Perform Installation
        try {
            require $basePath . '/vendor/autoload.php';
            $app = require_once $basePath . '/bootstrap/app.php';
            $kernel = $app->make(Illuminate\Contracts\Console\Kernel::class);
            $kernel->bootstrap();

            // 1. Update .env
            $db = $_SESSION['db_config'];
            $envContent = file_get_contents($basePath . '/.env.example'); // Use example as template
            $envContent = preg_replace('/DB_HOST=.*/', "DB_HOST=" . $db['host'], $envContent);
            $envContent = preg_replace('/DB_DATABASE=.*/', "DB_DATABASE=" . $db['name'], $envContent);
            $envContent = preg_replace('/DB_USERNAME=.*/', "DB_USERNAME=" . $db['user'], $envContent);
            $envContent = preg_replace('/DB_PASSWORD=.*/', "DB_PASSWORD=" . $db['pass'], $envContent);
            $envContent = preg_replace('/APP_ENV=.*/', "APP_ENV=production", $envContent);
            $envContent = preg_replace('/APP_DEBUG=.*/', "APP_DEBUG=false", $envContent);
            
            file_put_contents($envPath, $envContent);

            // 2. Generate Key
            \Illuminate\Support\Facades\Artisan::call('key:generate', ['--force' => true]);

            // 3. Run Migrations & Seeds
            \Illuminate\Support\Facades\Artisan::call('migrate:fresh', ['--force' => true, '--seed' => true]);

            // 4. Storage Link
            \Illuminate\Support\Facades\Artisan::call('storage:link');

            header('Location: install.php?step=4');
            exit;
        } catch (\Exception $e) {
            $errors[] = "Installation Failed: " . $e->getMessage();
        }
    }
}

?>

<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>SMARTONE ERP - Wizard Installer</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.rtl.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css" rel="stylesheet">
    <style>
        body { background: #0f172a; color: #fff; font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; min-height: 100vh; display: flex; align-items: center; justify-content: center; }
        .installer-container { max-width: 800px; width: 100%; padding: 20px; }
        .wizard-card { background: rgba(30, 41, 59, 0.7); backdrop-filter: blur(15px); border: 1px solid rgba(255,255,255,0.1); border-radius: 20px; box-shadow: 0 25px 50px rgba(0,0,0,0.5); overflow: hidden; }
        .wizard-header { background: linear-gradient(135deg, #6366f1, #a855f7); padding: 30px; text-align: center; }
        .wizard-body { padding: 40px; }
        .step-indicator { display: flex; justify-content: center; margin-bottom: 30px; }
        .step { width: 40px; height: 40px; border-radius: 50%; background: #334155; display: flex; align-items: center; justify-content: center; margin: 0 10px; font-weight: bold; position: relative; }
        .step.active { background: #6366f1; box-shadow: 0 0 15px rgba(99, 102, 241, 0.5); }
        .step.completed { background: #10b981; }
        .form-control { background: rgba(0,0,0,0.2); border: 1px solid #475569; color: #fff; border-radius: 10px; padding: 12px; }
        .form-control:focus { background: rgba(0,0,0,0.3); border-color: #6366f1; box-shadow: none; color: #fff; }
        .requirement-item { display: flex; justify-content: space-between; padding: 10px 0; border-bottom: 1px solid rgba(255,255,255,0.05); }
        .btn-wizard { padding: 12px 30px; border-radius: 12px; font-weight: 600; transition: transform 0.2s; }
        .btn-wizard:hover { transform: translateY(-2px); }
        .success-icon { font-size: 80px; color: #10b981; margin-bottom: 20px; }
    </style>
</head>
<body>

<div class="installer-container">
    <div class="text-center mb-4">
        <h2>SMARTONE <span style="color: #6366f1;">ERP</span></h2>
    </div>

    <div class="wizard-card">
        <div class="wizard-header">
            <h4 class="mb-0">مثبت النظام الذكي</h4>
            <p class="mb-0 small opacity-75">نظام إدارة المؤسسات المتكامل</p>
        </div>

        <div class="wizard-body">
            <!-- Step Indicators -->
            <div class="step-indicator">
                <div class="step <?= $step >= 1 ? ($step > 1 ? 'completed' : 'active') : '' ?>"><?= $step > 1 ? '<i class="fas fa-check"></i>' : '1' ?></div>
                <div class="step <?= $step >= 2 ? ($step > 2 ? 'completed' : 'active') : '' ?>"><?= $step > 2 ? '<i class="fas fa-check"></i>' : '2' ?></div>
                <div class="step <?= $step >= 3 ? ($step > 3 ? 'completed' : 'active') : '' ?>"><?= $step > 3 ? '<i class="fas fa-check"></i>' : '3' ?></div>
                <div class="step <?= $step >= 4 ? 'active' : '' ?>"><?= $step == 4 ? '<i class="fas fa-rocket"></i>' : '4' ?></div>
            </div>

            <?php if (!empty($errors)): ?>
                <div class="alert alert-danger mb-4">
                    <ul class="mb-0">
                        <?php foreach ($errors as $error): ?>
                            <li><?= htmlspecialchars($error) ?></li>
                        <?php endforeach; ?>
                    </ul>
                </div>
            <?php endif; ?>

            <?php if ($step === 1): ?>
                <!-- Step 1: Requirements -->
                <h5 class="mb-4 text-center">التحقق من متطلبات النظام</h5>
                
                <div class="mb-4">
                    <div class="requirement-item">
                        <span>إصدار PHP (>= <?= $requirements['php'] ?>)</span>
                        <span><?= checkPHPVersion($requirements['php']) ? '<i class="fas fa-check-circle text-success"></i>' : '<i class="fas fa-times-circle text-danger"></i>' ?></span>
                    </div>

                    <h6 class="mt-4 mb-2 text-muted">إضافات PHP المطلوبة:</h6>
                    <?php 
                    $extResults = checkExtensions($requirements['extensions']);
                    foreach ($extResults as $ext => $loaded): ?>
                        <div class="requirement-item">
                            <span>الإضافة: <?= $ext ?></span>
                            <span><?= $loaded ? '<i class="fas fa-check-circle text-success font-small"></i>' : '<i class="fas fa-times-circle text-danger"></i>' ?></span>
                        </div>
                    <?php endforeach; ?>

                    <h6 class="mt-4 mb-2 text-muted">تراخيص الملفات (Writable):</h6>
                    <?php 
                    $dirResults = checkPermissions($requirements['directories'], $basePath);
                    foreach ($dirResults as $dir => $writable): ?>
                        <div class="requirement-item">
                            <span>المجلد: <?= $dir ?></span>
                            <span><?= $writable ? '<i class="fas fa-check-circle text-success"></i>' : '<i class="fas fa-times-circle text-danger"></i>' ?></span>
                        </div>
                    <?php endforeach; ?>
                </div>

                <div class="text-center">
                    <a href="install.php?step=2" class="btn btn-wizard btn-primary">المتابعة للإعدادات <i class="fas fa-arrow-left ms-2"></i></a>
                </div>

            <?php elseif ($step === 2): ?>
                <!-- Step 2: Database -->
                <h5 class="mb-4 text-center">إعداد قاعدة البيانات</h5>
                <form method="POST">
                    <div class="mb-3">
                        <label class="form-label">خادم قاعدة البيانات (Host)</label>
                        <input type="text" name="db_host" class="form-control" value="localhost" required>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">اسم قاعدة البيانات (Database Name)</label>
                        <input type="text" name="db_name" class="form-control" placeholder="smartone_db" required>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">اسم المستخدم (Username)</label>
                        <input type="text" name="db_user" class="form-control" placeholder="root" required>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">كلمة المرور (Password)</label>
                        <input type="password" name="db_pass" class="form-control">
                    </div>
                    <div class="text-center mt-4">
                        <button type="submit" class="btn btn-wizard btn-primary">اختبار الاتصال والمتابعة <i class="fas fa-plug ms-2"></i></button>
                    </div>
                </form>

            <?php elseif ($step === 3): ?>
                <!-- Step 3: Processing -->
                <div class="text-center">
                    <div class="spinner-border text-primary mb-4" style="width: 3rem; height: 3rem;" role="status"></div>
                    <h5>جاري تثبيت النظام...</h5>
                    <p class="text-muted">نحن نقوم الآن بإنشاء الجداول، تحميل البيانات الأولية، وإعداد ملفات النظام.</p>
                    
                    <form method="POST" id="installForm">
                        <input type="hidden" name="action" value="install">
                        <button type="submit" class="btn btn-wizard btn-success btn-lg">ابدأ التثبيت الآن <i class="fas fa-check-double ms-2"></i></button>
                    </form>
                </div>

            <?php elseif ($step === 4): ?>
                <!-- Step 4: Success -->
                <div class="text-center">
                    <i class="fas fa-check-circle success-icon"></i>
                    <h3 class="mb-3">مبروك! تم التثبيت بنجاح</h3>
                    <p class="mb-4 text-muted">النظام الآن جاهز للعمل. يرجى التأكد من حذف ملف التثبيت لدواعي أمنية.</p>
                    
                    <div class="d-grid gap-3 col-md-6 mx-auto">
                        <a href="/" class="btn btn-wizard btn-primary">الدخول للنظام <i class="fas fa-external-link-alt ms-2"></i></a>
                    </div>

                    <div class="mt-5 p-3 bg-danger bg-opacity-10 border border-danger rounded text-danger">
                        <i class="fas fa-exclamation-triangle me-2"></i> 
                        <strong>هام جداً:</strong> قم بحذف ملف <code>public/install.php</code> فوراً من الخادم.
                    </div>
                </div>
            <?php endif; ?>
        </div>
    </div>
    
    <div class="text-center mt-4 text-muted small">
        &copy; <?= date('Y') ?> SMARTONE ERP. جميع الحقوق محفوظة.
    </div>
</div>

</body>
</html>
