@extends('layouts.app')

@section('content')
    <div class="d-flex justify-content-between align-items-center mb-4">
        <div>
            <h1 class="page-title">{{ __('Edit Quotation #') }}{{ $quotation->quotation_number }}</h1>
            <nav aria-label="breadcrumb">
                <ol class="breadcrumb">
                    <li class="breadcrumb-item"><a
                            href="{{ route('accounting.quotations.index') }}">{{ __('Quotations') }}</a></li>
                    <li class="breadcrumb-item active">{{ __('Edit') }}</li>
                </ol>
            </nav>
        </div>
    </div>

    <form action="{{ route('accounting.quotations.update', $quotation) }}" method="POST" id="quotationForm">
        @csrf
        @method('PUT')
        <div class="row">
            <div class="col-lg-8">
                <div class="card mb-4">
                    <div class="card-body">
                        <div class="row g-3 mb-4">
                            <div class="col-md-6">
                                <label class="form-label">{{ __('Customer') }}</label>
                                <select name="customer_id" class="form-select" required>
                                    <option value="">{{ __('Select Customer') }}</option>
                                    @foreach ($customers as $customer)
                                        <option value="{{ $customer->id }}"
                                            {{ $quotation->customer_id == $customer->id ? 'selected' : '' }}>
                                            {{ $customer->name }}</option>
                                    @endforeach
                                </select>
                            </div>
                            <div class="col-md-2">
                                <label class="form-label">{{ __('Currency') }}</label>
                                <select name="currency_id" id="currencySelect" class="form-select" disabled>
                                    <option value="{{ $quotation->currency_id }}">
                                        {{ $quotation->currency->code ?? 'EGP' }}
                                    </option>
                                </select>
                                <input type="hidden" name="currency_id" value="{{ $quotation->currency_id }}">
                            </div>
                            <div class="col-md-2">
                                <label class="form-label">{{ __('Exchange Rate') }}</label>
                                <input type="number" name="exchange_rate" id="exchangeRateInput" class="form-control"
                                    step="0.000001" min="0" value="{{ $quotation->exchange_rate }}" readonly>
                            </div>
                            <div class="col-md-3">
                                <label class="form-label">{{ __('Date') }}</label>
                                <input type="date" name="date" class="form-control"
                                    value="{{ $quotation->date->format('Y-m-d') }}" required>
                            </div>
                            <div class="col-md-3">
                                <label class="form-label">{{ __('Expiry Date') }}</label>
                                <input type="date" name="expiry_date" class="form-control"
                                    value="{{ $quotation->expiry_date ? $quotation->expiry_date->format('Y-m-d') : '' }}">
                            </div>
                        </div>

                        <h5 class="mb-3">{{ __('Items') }}</h5>
                        <div class="table-responsive">
                            <table class="table table-bordered" id="itemsTable">
                                <thead>
                                    <tr>
                                        <th style="width: 40%">{{ __('Description') }}</th>
                                        <th style="width: 15%">{{ __('Qty') }}</th>
                                        <th style="width: 20%">{{ __('Unit Price') }}</th>
                                        <th style="width: 20%">{{ __('Total') }}</th>
                                        <th style="width: 5%"></th>
                                    </tr>
                                </thead>
                                <tbody>
                                    @foreach ($quotation->items as $index => $item)
                                        <tr class="item-row">
                                            <td>
                                                <select name="items[{{ $index }}][product_id]"
                                                    class="form-select product-select" onchange="fillProductData(this)">
                                                    <option value="">{{ __('Select Product') }}</option>
                                                    @foreach ($products as $product)
                                                        <option value="{{ $product->id }}"
                                                            data-name="{{ $product->name }}"
                                                            data-price="{{ $product->price }}"
                                                            data-price-usd="{{ $product->price_usd ?? 0 }}"
                                                            data-description="{{ $product->description }}"
                                                            data-unit-id="{{ $product->unit_id }}"
                                                            {{ $item->product_id == $product->id ? 'selected' : '' }}>
                                                            {{ $product->name }}
                                                            ({{ number_format($product->price, 2) }}
                                                            {{ __('EGP') }} /
                                                            {{ $product->price_usd ? number_format($product->price_usd, 2) . ' USD' : '-' }})
                                                        </option>
                                                    @endforeach
                                                </select>
                                                <textarea name="items[{{ $index }}][description]" class="form-control mt-1 description-input"
                                                    placeholder="{{ __('Description') }}" rows="2">{{ $item->description }}</textarea>
                                            </td>
                                            <td>
                                                <input type="number" name="items[{{ $index }}][quantity]"
                                                    class="form-control quantity" value="{{ $item->quantity + 0 }}"
                                                    min="0.01" step="0.01" required>
                                            </td>
                                            <td>
                                                <input type="number" name="items[{{ $index }}][unit_price]"
                                                    class="form-control price" value="{{ $item->unit_price }}"
                                                    min="0" step="0.01" required>
                                            </td>
                                            <td>
                                                <input type="number" class="form-control total"
                                                    value="{{ number_format($item->total, 2, '.', '') }}" readonly>
                                            </td>
                                            <td class="text-center">
                                                <button type="button" class="btn btn-sm btn-danger remove-row"><i
                                                        class="fas fa-trash"></i></button>
                                            </td>
                                        </tr>
                                    @endforeach
                                </tbody>
                            </table>
                        </div>
                        <button type="button" class="btn btn-sm btn-secondary" id="addItem">
                            <i class="fas fa-plus me-1"></i> {{ __('Add Item') }}
                        </button>
                    </div>
                </div>

                <div class="card">
                    <div class="card-body">
                        <div class="mb-3">
                            <label class="form-label">{{ __('Notes') }}</label>
                            <textarea name="notes" class="form-control" rows="3">{{ $quotation->notes }}</textarea>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">{{ __('Terms & Conditions') }}</label>
                            <textarea name="terms" class="form-control" rows="3">{{ $quotation->terms }}</textarea>
                        </div>
                    </div>
                </div>
            </div>

            <div class="col-lg-4">
                <div class="card mb-4">
                    <div class="card-body">
                        <h5 class="card-title mb-4">{{ __('Summary') }}</h5>
                        <div class="d-flex justify-content-between mb-2">
                            <span>{{ __('Subtotal:') }}</span>
                            <span id="subtotalDisplay">{{ number_format($quotation->subtotal, 2) }}</span>
                        </div>
                        <hr>
                        <div class="d-flex justify-content-between mb-2 align-items-center">
                            <span>{{ __('Discount:') }}</span>
                            <div class="input-group input-group-sm w-50">
                                <select name="discount_type" id="discountType" class="form-select">
                                    <option value="fixed" {{ $quotation->discount_type == 'fixed' ? 'selected' : '' }}>{{ __('Fixed') }}</option>
                                    <option value="percent" {{ $quotation->discount_type == 'percent' ? 'selected' : '' }}>{{ __('%') }}</option>
                                </select>
                                <input type="number" name="discount_value" id="discountInput" class="form-control" 
                                    value="{{ $quotation->discount_value + 0 }}" min="0" step="0.01">
                            </div>
                        </div>
                        <div class="d-flex justify-content-between mb-2 align-items-center">
                            <span>{{ __('Shipping:') }}</span>
                            <input type="number" name="shipping_amount" id="shippingInput" class="form-control form-control-sm w-50 text-end" 
                                value="{{ $quotation->shipping_amount + 0 }}" min="0" step="0.01">
                        </div>
                        <div class="d-flex justify-content-between mb-2 align-items-center">
                            <span>{{ __('Extra Exp:') }}</span>
                            <div class="input-group input-group-sm w-50">
                                <input type="text" name="extra_expenses_note" class="form-control" placeholder="Note" 
                                    value="{{ $quotation->extra_expenses_note }}">
                                <input type="number" name="extra_expenses_amount" id="extraExpensesInput" class="form-control text-end" 
                                    value="{{ $quotation->extra_expenses_amount + 0 }}" min="0" step="0.01">
                            </div>
                        </div>
                        <hr>
                        <div class="d-flex justify-content-between">
                            <span class="fw-bold">{{ __('Grand Total:') }}</span>
                            <span class="fw-bold fs-5" id="grandTotalDisplay">{{ number_format($quotation->grand_total, 2) }}</span>
                        </div>
                        <hr>
                        <button type="submit" class="btn btn-primary w-100 mb-2">
                            <i class="fas fa-save me-2"></i>{{ __('Update Quotation') }}
                        </button>
                        <a href="{{ route('accounting.quotations.index') }}"
                            class="btn btn-outline-light w-100">{{ __('Cancel') }}</a>
                    </div>
                </div>
            </div>
        </div>
    </form>

    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script>
        $(document).ready(function() {
            let rowCount = {{ $quotation->items->count() }};

            function calculateRow(row) {
                const qty = parseFloat(row.find('.quantity').val()) || 0;
                const price = parseFloat(row.find('.price').val()) || 0;
                const total = qty * price;
                row.find('.total').val(total.toFixed(2));
                calculateTotals();
            }

            function calculateTotals() {
                let subtotal = 0;
                $('.item-row').each(function() {
                    subtotal += parseFloat($(this).find('.total').val()) || 0;
                });

                // Discount
                const discountType = document.getElementById('discountType').value;
                let discountValue = parseFloat(document.getElementById('discountInput').value) || 0;
                let discountAmount = 0;

                if (discountType === 'percent') {
                    discountAmount = (subtotal * discountValue) / 100;
                } else {
                    discountAmount = discountValue;
                }

                // Expenses
                const shippingAmount = parseFloat(document.getElementById('shippingInput').value) || 0;
                const extraExpensesAmount = parseFloat(document.getElementById('extraExpensesInput').value) || 0;

                const grandTotal = subtotal - discountAmount + shippingAmount + extraExpensesAmount;

                $('#subtotalDisplay').text(subtotal.toFixed(2));
                $('#grandTotalDisplay').text(grandTotal.toFixed(2));
            }
            
            // Event Listeners for Calculation
            $('#discountInput, #shippingInput, #extraExpensesInput').on('input', calculateTotals);
            $('#discountType').change(calculateTotals);

            $('#addItem').click(function() {
                const newRow = `
                            <tr class="item-row">
                                <td>
                                    <select name="items[${rowCount}][product_id]" class="form-select product-select" onchange="fillProductData(this)">
                                        <option value="">{{ __('Select Product') }}</option>
                                        @foreach ($products as $product)
                                            <option value="{{ $product->id }}" data-name="{{ $product->name }}"
                                                data-price="{{ $product->price }}"
                                                data-price-usd="{{ $product->price_usd ?? 0 }}"
                                                data-description="{{ $product->description }}"
                                                data-unit-id="{{ $product->unit_id }}">
                                                {{ $product->name }}
                                            </option>
                                        @endforeach
                                    </select>
                                    <textarea name="items[${rowCount}][description]" class="form-control mt-1 description-input" placeholder="{{ __('Description') }}" rows="2"></textarea>
                                </td>
                                <td>
                                    <input type="number" name="items[${rowCount}][quantity]" class="form-control quantity" value="1" min="1" step="0.01" required>
                                </td>
                                <td>
                                    <input type="number" name="items[${rowCount}][unit_price]" class="form-control price" value="0.00" min="0" step="0.01" required>
                                </td>
                                <td>
                                    <input type="number" class="form-control total" value="0.00" readonly>
                                </td>
                                <td class="text-center">
                                    <button type="button" class="btn btn-sm btn-danger remove-row"><i class="fas fa-trash"></i></button>
                                </td>
                            </tr>
                        `;
                $('#itemsTable tbody').append(newRow);
                rowCount++;
            });

            $(document).on('click', '.remove-row', function() {
                if ($('.item-row').length > 1) {
                    $(this).closest('tr').remove();
                    calculateTotals();
                }
            });

            $(document).on('input', '.quantity, .price', function() {
                calculateRow($(this).closest('tr'));
            });
        });

        function fillProductData(selectElement) {
            const selectedOption = selectElement.options[selectElement.selectedIndex];
            const row = selectElement.closest('.item-row');
            
            // For Edit View, we rely on the saved Exchange Rate/Currency so we don't need complex currency logic
            // But we should check if quote is in USD or EGP effectively.
            // Since currency_id is fixed in edit (disabled), we can check it.
            const currencyCode = "{{ $quotation->currency->code ?? 'EGP' }}";
            let useUsd = false;
            
            if (currencyCode.includes('USD') || currencyCode.includes('Dollar')) {
                useUsd = true;
            }

            if (selectedOption.value) {
                const productName = selectedOption.getAttribute('data-name');
                const productPrice = selectedOption.getAttribute('data-price');
                const productPriceUsd = selectedOption.getAttribute('data-price-usd');
                const productDescription = selectedOption.getAttribute('data-description');

                // Fill description
                row.querySelector('.description-input').value = productDescription || productName;

                // Fill price
                 let finalPrice = parseFloat(productPrice);
                if (useUsd && parseFloat(productPriceUsd) > 0) {
                    finalPrice = parseFloat(productPriceUsd);
                }

                row.querySelector('.price').value = finalPrice.toFixed(2);

                // Trigger calculation
                const priceInput = row.querySelector('.price');
                const event = new Event('input', {
                    bubbles: true
                });
                priceInput.dispatchEvent(event);
            }
        }
    </script>
@endsection
